<?php
// Inicia a sessão
session_start();
include 'conexao.php'; // Certifique-se de que 'conexao.php' está correto

// Verificação de sessão e perfil
$usuario_id = (int)($_SESSION['usuario']['id'] ?? 0);
$perfil = $_SESSION['usuario']['perfil'] ?? '';

if (strtoupper(trim($perfil)) !== 'PROFESSOR' || $usuario_id <= 0) {
    header('Location: login.php');
    exit;
}

// Nome do professor
$nome_professor = '';
$sqlNome = "SELECT nome FROM usuarios WHERE id = ? LIMIT 1";
$stmt = $conn->prepare($sqlNome);
if ($stmt) {
    $stmt->bind_param("i", $usuario_id);
    $stmt->execute();
    $resNome = $stmt->get_result();
    if ($resNome && $resNome->num_rows === 1) {
        $rowNome = $resNome->fetch_assoc();
        $nome_professor = $rowNome['nome'];
    }
    $stmt->close();
}

// Ano lectivo
$ano_lectivo_atual = '2025/2026';

// ------------------------------------------
// 1. LÓGICA DE SALVAR AVALIAÇÕES CONTÍNUAS (AVs) E MAC
// ------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['salvar_avs'])) {
    $turma_id = (int)($_POST['turma_id'] ?? 0);
    $disciplina_id = (int)($_POST['disciplina_id'] ?? 0);
    $trimestre_selecionado = (int)($_POST['trimestre'] ?? 0);

    // [Omissão: Verifica se o prazo foi fechado, assim como no seu código original]

    $conn->begin_transaction();
    try {
        // Tabela `avaliacoes_continuas` - ADICIONADO OS CAMPOS DE FALTA
        $stmt_insert_or_update_avs = $conn->prepare("
            INSERT INTO avaliacoes_continuas 
            (aluno_id, disciplina_id, turma_id, trimestre, av1, av2, av3, av4, av5, av6, falta_av1, falta_av2, falta_av3, falta_av4, falta_av5, falta_av6, mac, ano_lectivo)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE 
                av1=VALUES(av1), av2=VALUES(av2), av3=VALUES(av3), av4=VALUES(av4), av5=VALUES(av5), av6=VALUES(av6), 
                falta_av1=VALUES(falta_av1), falta_av2=VALUES(falta_av2), falta_av3=VALUES(falta_av3), falta_av4=VALUES(falta_av4), falta_av5=VALUES(falta_av5), falta_av6=VALUES(falta_av6),
                mac=VALUES(mac)
        ");

        foreach ($_POST['aluno_id'] as $aluno_id) {
            $aluno_id = (int)$aluno_id;
            
            // Lendo todas as 6 AVs do POST
            $av1 = (float)($_POST["av1_{$aluno_id}"] ?? 0);
            $av2 = (float)($_POST["av2_{$aluno_id}"] ?? 0);
            $av3 = (float)($_POST["av3_{$aluno_id}"] ?? 0);
            $av4 = (float)($_POST["av4_{$aluno_id}"] ?? 0);
            $av5 = (float)($_POST["av5_{$aluno_id}"] ?? 0);
            $av6 = (float)($_POST["av6_{$aluno_id}"] ?? 0);
            
            // LENDO OS NOVOS CAMPOS DE FALTA (Checkbox: 'on' se marcado, '' se não, convertemos para 1 ou 0)
            $falta_av1 = isset($_POST["falta_av1_{$aluno_id}"]) ? 1 : 0;
            $falta_av2 = isset($_POST["falta_av2_{$aluno_id}"]) ? 1 : 0;
            $falta_av3 = isset($_POST["falta_av3_{$aluno_id}"]) ? 1 : 0;
            $falta_av4 = isset($_POST["falta_av4_{$aluno_id}"]) ? 1 : 0;
            $falta_av5 = isset($_POST["falta_av5_{$aluno_id}"]) ? 1 : 0;
            $falta_av6 = isset($_POST["falta_av6_{$aluno_id}"]) ? 1 : 0;

            $mac = 0.00;
            
            // Array com as avaliações
            $avaliacoes = [$av1, $av2, $av3, $av4, $av5, $av6];
            $soma_avs = 0;
            $contagem_avs = 0;

            // 1. Calcula a soma e a contagem de AVs válidas (maiores que zero)
            foreach ($avaliacoes as $av) {
                if ($av > 0) {
                    $soma_avs += $av;
                    $contagem_avs++;
                }
            }

            // 2. REGRA AJUSTADA: Calcula o MAC SE o número de AVs lançadas for >= 3
            if ($contagem_avs >= 3) {
                $mac = $soma_avs / $contagem_avs; 
            }
            
            // Arredonda o MAC para duas casas decimais
            $mac = round($mac, 2);

            // 18 TIPOS para 18 VARIÁVEIS (4x 'i' + 6x 'd' + 6x 'i' + 1x 'd' + 1x 's')
            $stmt_insert_or_update_avs->bind_param(
                "iiiiddddddiiddddss", 
                $aluno_id, $disciplina_id, $turma_id, $trimestre_selecionado,
                $av1, $av2, $av3, $av4, $av5, $av6, 
                $falta_av1, $falta_av2, $falta_av3, $falta_av4, $falta_av5, $falta_av6, // Novos campos
                $mac, $ano_lectivo_atual
            );
            $stmt_insert_or_update_avs->execute();

            // Atualiza a coluna MAC na tabela `notas`
            $stmt_update_notas = $conn->prepare("
                INSERT INTO notas 
                (aluno_id, disciplina_id, turma_id, trimestre, mac, ano_lectivo)
                VALUES (?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE 
                    mac=VALUES(mac)
            ");
            $stmt_update_notas->bind_param(
                "iiiids",
                $aluno_id, $disciplina_id, $turma_id, $trimestre_selecionado,
                $mac, $ano_lectivo_atual
            );
            $stmt_update_notas->execute();
            $stmt_update_notas->close();
        }

        $stmt_insert_or_update_avs->close();
        $conn->commit();
        $_SESSION['mensagem_sucesso'] = "Avaliações Contínuas, faltas e MAC salvas com sucesso!";
    } catch (mysqli_sql_exception $e) {
        $conn->rollback();
        $_SESSION['mensagem_erro'] = "Erro ao salvar: " . $e->getMessage();
    }
    header("Location: caderneta_professor.php?turma_id={$turma_id}&disciplina_id={$disciplina_id}&trimestre={$trimestre_selecionado}");
    exit;
}

// ------------------------------------------
// 2. FILTROS E BUSCA DE DADOS (Inalterado)
// ------------------------------------------
$turma_id = $_GET['turma_id'] ?? null;
$disciplina_id = $_GET['disciplina_id'] ?? null;
$trimestre_selecionado = $_GET['trimestre'] ?? null;
$nome_disciplina = '';
$nome_turma_selecionada = '';

// Turmas do professor
$turmas = [];
$sqlTurmas = "
    SELECT DISTINCT t.id, t.nome, t.classe, t.sala, t.curso, t.turno
    FROM turmas t
    INNER JOIN atribuicoes a ON t.id = a.turma_id
    WHERE a.professor_id = ?
    ORDER BY t.classe, t.nome
";
$stmt = $conn->prepare($sqlTurmas);
$stmt->bind_param("i", $usuario_id);
$stmt->execute();
$turmas_res = $stmt->get_result();
while ($row = $turmas_res->fetch_assoc()) {
    $partes = [];
    if ($row['classe']) $partes[] = $row['classe'];
    if ($row['nome']) $partes[] = $row['nome'];
    if ($row['curso']) $partes[] = $row['curso'];
    if ($row['turno']) $partes[] = $row['turno'];
    if ($row['sala']) $partes[] = $row['sala'];
    $row['descricao_formatada'] = implode('-', $partes);
    $turmas[] = $row;
}
$stmt->close();

// Disciplinas
$disciplinas = [];
if ($turma_id) {
    $sqlDisciplinas = "
        SELECT DISTINCT d.id, d.nome
        FROM atribuicoes a
        INNER JOIN disciplinas d ON a.disciplina_id = d.id
        WHERE a.professor_id = ? AND a.turma_id = ?
        ORDER BY d.nome
    ";
    $stmt = $conn->prepare($sqlDisciplinas);
    $stmt->bind_param("ii", $usuario_id, $turma_id);
    $stmt->execute();
    $disciplinas_res = $stmt->get_result();
    while ($disciplina = $disciplinas_res->fetch_assoc()) {
        $disciplinas[] = $disciplina;
    }
    $stmt->close();
}

// Nome da disciplina selecionada
if ($disciplina_id) {
    $sqlNomeDisciplina = "SELECT nome FROM disciplinas WHERE id = ?";
    $stmt = $conn->prepare($sqlNomeDisciplina);
    $stmt->bind_param("i", $disciplina_id);
    $stmt->execute();
    $res = $stmt->get_result();
    if ($res->num_rows > 0) {
        $row = $res->fetch_assoc();
        $nome_disciplina = $row['nome'];
    }
    $stmt->close();
}

// Nome da turma
if ($turma_id) {
    foreach ($turmas as $turma) {
        if ($turma['id'] == $turma_id) {
            $nome_turma_selecionada = $turma['descricao_formatada'];
            break;
        }
    }
}

// Alunos e avaliações
$alunos = [];
$avs_por_aluno = [];
$prazo_fechado = false; // Adicionar lógica de fecho se necessário
if ($turma_id && $disciplina_id && $trimestre_selecionado) {
    // Buscar alunos da turma
    $sqlAlunos = "SELECT id, nome FROM alunos WHERE turma_id = ? ORDER BY nome";
    $stmtAlunos = $conn->prepare($sqlAlunos);
    $stmtAlunos->bind_param("i", $turma_id);
    $stmtAlunos->execute();
    $alunos_res = $stmtAlunos->get_result();
    $alunos = $alunos_res->fetch_all(MYSQLI_ASSOC);
    $stmtAlunos->close();

    // Buscar avaliações já lançadas (SELECIONANDO TAMBÉM AS NOVAS COLUNAS DE FALTA)
    $aluno_ids = array_column($alunos, 'id');
    if (!empty($aluno_ids)) {
        $in_clause = str_repeat('?,', count($aluno_ids) - 1) . '?';
        $sqlAVs = "SELECT *, falta_av1, falta_av2, falta_av3, falta_av4, falta_av5, falta_av6 FROM avaliacoes_continuas 
            WHERE aluno_id IN ($in_clause) AND disciplina_id = ? AND turma_id = ? AND trimestre = ?";
        $params_types = str_repeat('i', count($aluno_ids)) . 'iii';
        $params = array_merge($aluno_ids, [$disciplina_id, $turma_id, $trimestre_selecionado]);

        $stmtAVs = $conn->prepare($sqlAVs);
        $stmtAVs->bind_param($params_types, ...$params);
        $stmtAVs->execute();
        $resultAVs = $stmtAVs->get_result();
        while ($av = $resultAVs->fetch_assoc()) {
            $avs_por_aluno[$av['aluno_id']] = $av;
        }
        $stmtAVs->close();
    }
}

// Mensagens
$mensagem_sucesso = $_SESSION['mensagem_sucesso'] ?? '';
$mensagem_erro = $_SESSION['mensagem_erro'] ?? '';
unset($_SESSION['mensagem_sucesso'], $_SESSION['mensagem_erro']);
?>
<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Caderneta do Professor - Avaliações Contínuas</title>
    
    <link rel="stylesheet" href="css/all.min.css">
    <link rel="stylesheet" href="css/inter.css">
    <script src="js/tailwindcss.js"></script>
    <style>
        /* ... (Estilos CSS do seu arquivo original) ... */
        .sidebar { transition: transform 0.3s; transform: translateX(-100%); z-index: 999; }
        .sidebar.active { transform: translateX(0); }
        .overlay { position: fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.5); z-index:998; display:none; }
        .overlay.active { display:block; }
        @media (min-width: 768px) {
            .sidebar { transform: translateX(0); }
            .content { margin-left: 16rem; }
            .overlay { display: none !important; }
        }
        input[type="number"]::-webkit-outer-spin-button,
        input[type="number"]::-webkit-inner-spin-button { -webkit-appearance: none; margin: 0; }
        input[type="number"] { -moz-appearance: textfield; }
        .nota-av-input {
            width: 40px; /* Tamanho reduzido para mobile para caber 6 AVs */
            font-size: 0.75rem;
        }
        @media (min-width: 768px) {
            .nota-av-input {
                width: 60px; /* Tamanho padrão em desktop */
                font-size: 0.9rem;
            }
        }
        /* Ajuste para caber as 12 colunas de Nota/Falta + Aluno + Num + MAC */
        .w-full.border.border-gray-300.text-sm {
            min-width: 900px; 
        }
    </style>
</head>
<body class="bg-gray-100 min-h-screen flex flex-col">

<header class="fixed top-0 left-0 w-full md:left-64 md:w-[calc(100%-16rem)] h-16 bg-blue-900 text-white flex items-center justify-between p-4 shadow-lg z-50">
    <button id="menu-toggle" class="md:hidden text-2xl"><i class="fas fa-bars"></i></button>
    <span class="font-bold text-lg sm:text-xl">Caderneta - Lançamento AVs</span>
    <div class="flex items-center gap-4 ml-auto">
        <span class="hidden sm:block font-bold"><?= htmlspecialchars($nome_professor) ?></span>
    </div>
</header>

<div id="overlay" class="overlay"></div>

<nav id="sidebar" class="sidebar fixed top-0 left-0 w-64 h-full bg-blue-900 text-white p-4 pt-16 shadow-lg overflow-y-auto">
    <div class="flex flex-col items-center justify-center p-4">
        <span class="text-xl font-bold text-center"><?= htmlspecialchars($nome_professor) ?></span>
    </div>
    <ul class="mt-8 space-y-2">
        <li><a href="painel_professor1.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700"><i class="fas fa-arrow-left"></i> <span>Voltar ao Painel</span></a></li>
        <li><a href="caderneta_professor.php" class="flex items-center gap-3 p-3 rounded-lg bg-blue-700"><i class="fas fa-book"></i> <span>Caderneta (AVs)</span></a></li>
        <li><a href="mini_pauta_professor1.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700"><i class="fas fa-pen"></i> <span>Mini-Pauta (MT)</span></a></li>
        <li><a href="login.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700"><i class="fas fa-sign-out-alt"></i> <span>Sair</span></a></li>
    </ul>
</nav>

<main class="content flex-grow p-4 md:p-6 md:ml-64 mt-16">
    <h1 class="text-3xl font-bold text-gray-800 mb-6">Caderneta: Lançamento de Avaliações Contínuas (AVs)</h1>

    <?php if ($mensagem_sucesso): ?>
        <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4"><?= htmlspecialchars($mensagem_sucesso) ?></div>
    <?php endif; ?>
    <?php if ($mensagem_erro): ?>
        <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4"><?= htmlspecialchars($mensagem_erro) ?></div>
    <?php endif; ?>

    <form action="" method="GET" class="bg-white p-6 rounded-xl shadow-md mb-6 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 items-end">
        <div>
            <label for="turma" class="block text-gray-700 font-semibold mb-2">Turma:</label>
            <select id="turma" name="turma_id" class="w-full p-3 rounded-lg border border-gray-300 focus:ring focus:ring-blue-200 transition-colors" required onchange="this.form.submit()">
                <option value="">Selecione</option>
                <?php foreach ($turmas as $turma): ?>
                    <option value="<?= $turma['id'] ?>" <?= $turma_id == $turma['id'] ? 'selected' : '' ?>><?= $turma['descricao_formatada'] ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div>
            <label for="disciplina" class="block text-gray-700 font-semibold mb-2">Disciplina:</label>
            <select id="disciplina" name="disciplina_id" class="w-full p-3 rounded-lg border border-gray-300 focus:ring focus:ring-blue-200 transition-colors" required>
                <option value="">Selecione</option>
                <?php foreach ($disciplinas as $disciplina): ?>
                    <option value="<?= $disciplina['id'] ?>" <?= $disciplina_id == $disciplina['id'] ? 'selected' : '' ?>><?= $disciplina['nome'] ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div>
            <label for="trimestre" class="block text-gray-700 font-semibold mb-2">Trimestre:</label>
            <select id="trimestre" name="trimestre" class="w-full p-3 rounded-lg border border-gray-300 focus:ring focus:ring-blue-200 transition-colors" required>
                <option value="">Selecione</option>
                <option value="1" <?= $trimestre_selecionado == 1 ? 'selected' : '' ?>>1º</option>
                <option value="2" <?= $trimestre_selecionado == 2 ? 'selected' : '' ?>>2º</option>
                <option value="3" <?= $trimestre_selecionado == 3 ? 'selected' : '' ?>>3º</option>
            </select>
        </div>
        <div class="col-span-1 sm:col-span-2 lg:col-span-1">
            <button type="submit" class="w-full bg-blue-900 text-white py-3 px-6 rounded-lg hover:bg-blue-800 transition"><i class="fas fa-search"></i> Filtrar</button>
        </div>
    </form>

    <?php if ($turma_id && $disciplina_id && $trimestre_selecionado): ?>
        <div class="bg-white p-6 rounded-xl shadow-md">
            <h2 class="text-xl font-bold text-gray-700 mb-4"><?= htmlspecialchars($nome_disciplina) ?> - <?= htmlspecialchars($nome_turma_selecionada) ?> (<?= $trimestre_selecionado ?>º Trimestre)</h2>

            <form method="POST">
                <input type="hidden" name="turma_id" value="<?= $turma_id ?>">
                <input type="hidden" name="disciplina_id" value="<?= $disciplina_id ?>">
                <input type="hidden" name="trimestre" value="<?= $trimestre_selecionado ?>">

                <div class="overflow-x-auto">
                    <table class="w-full border border-gray-300 text-sm">
                        <thead class="bg-blue-900 text-white">
                            <tr>
                                <th class="border px-2 py-1" rowspan="2">Nº</th>
                                <th class="border px-2 py-1 text-left whitespace-nowrap" rowspan="2">Nome do Aluno</th>
                                <th colspan="2" class="border px-2 py-1">AV1</th> 
                                <th colspan="2" class="border px-2 py-1">AV2</th> 
                                <th colspan="2" class="border px-2 py-1">AV3</th> 
                                <th colspan="2" class="border px-2 py-1">AV4</th> 
                                <th colspan="2" class="border px-2 py-1">AV5</th> 
                                <th colspan="2" class="border px-2 py-1">AV6</th> 
                                <th rowspan="2" class="border px-2 py-1 align-middle">MAC</th>
                            </tr>
                            <tr>
                                <th class="border px-2 py-1">Nota</th>
                                <th class="border px-2 py-1">F</th>
                                <th class="border px-2 py-1">Nota</th>
                                <th class="border px-2 py-1">F</th>
                                <th class="border px-2 py-1">Nota</th>
                                <th class="border px-2 py-1">F</th>
                                <th class="border px-2 py-1">Nota</th>
                                <th class="border px-2 py-1">F</th>
                                <th class="border px-2 py-1">Nota</th>
                                <th class="border px-2 py-1">F</th>
                                <th class="border px-2 py-1">Nota</th>
                                <th class="border px-2 py-1">F</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($alunos)): ?>
                                <tr><td colspan="15" class="text-center py-4 text-gray-500">Nenhum aluno encontrado.</td></tr>
                            <?php else: ?>
                                <?php $i = 1; foreach ($alunos as $aluno):
                                    $avs = $avs_por_aluno[$aluno['id']] ?? [];
                                    $av1 = $avs['av1'] ?? '';
                                    $av2 = $avs['av2'] ?? '';
                                    $av3 = $avs['av3'] ?? '';
                                    $av4 = $avs['av4'] ?? '';
                                    $av5 = $avs['av5'] ?? ''; 
                                    $av6 = $avs['av6'] ?? ''; 
                                    // Novas variáveis para falta
                                    $falta_av1 = $avs['falta_av1'] ?? 0; 
                                    $falta_av2 = $avs['falta_av2'] ?? 0; 
                                    $falta_av3 = $avs['falta_av3'] ?? 0; 
                                    $falta_av4 = $avs['falta_av4'] ?? 0; 
                                    $falta_av5 = $avs['falta_av5'] ?? 0; 
                                    $falta_av6 = $avs['falta_av6'] ?? 0; 
                                    $mac = $avs['mac'] ?? '';
                                ?>
                                    <tr>
                                        <td class="border px-2 py-1 text-center"><?= $i++ ?></td>
                                        <td class="border px-2 py-1 whitespace-nowrap"><?= htmlspecialchars($aluno['nome']) ?></td>
                                        
                                        <td class="border px-1 py-1">
                                            <input type="number" step="0.01" name="av1_<?= $aluno['id'] ?>" 
                                                value="<?= $av1 ?>" class="nota-av-input w-full border p-1 text-center" 
                                                data-aluno-id="<?= $aluno['id'] ?>" min="0" max="20" <?= $prazo_fechado ? 'readonly' : '' ?>>
                                        </td>
                                        <td class="border px-1 py-1 text-center">
                                            <input type="checkbox" name="falta_av1_<?= $aluno['id'] ?>" value="1" 
                                                <?= $falta_av1 == 1 ? 'checked' : '' ?> <?= $prazo_fechado ? 'disabled' : '' ?> class="h-4 w-4 text-red-600 border-gray-300 rounded focus:ring-red-500">
                                        </td>
                                        
                                        <td class="border px-1 py-1">
                                            <input type="number" step="0.01" name="av2_<?= $aluno['id'] ?>" 
                                                value="<?= $av2 ?>" class="nota-av-input w-full border p-1 text-center" 
                                                data-aluno-id="<?= $aluno['id'] ?>" min="0" max="20" <?= $prazo_fechado ? 'readonly' : '' ?>>
                                        </td>
                                        <td class="border px-1 py-1 text-center">
                                            <input type="checkbox" name="falta_av2_<?= $aluno['id'] ?>" value="1" 
                                                <?= $falta_av2 == 1 ? 'checked' : '' ?> <?= $prazo_fechado ? 'disabled' : '' ?> class="h-4 w-4 text-red-600 border-gray-300 rounded focus:ring-red-500">
                                        </td>

                                        <td class="border px-1 py-1">
                                            <input type="number" step="0.01" name="av3_<?= $aluno['id'] ?>" 
                                                value="<?= $av3 ?>" class="nota-av-input w-full border p-1 text-center" 
                                                data-aluno-id="<?= $aluno['id'] ?>" min="0" max="20" <?= $prazo_fechado ? 'readonly' : '' ?>>
                                        </td>
                                        <td class="border px-1 py-1 text-center">
                                            <input type="checkbox" name="falta_av3_<?= $aluno['id'] ?>" value="1" 
                                                <?= $falta_av3 == 1 ? 'checked' : '' ?> <?= $prazo_fechado ? 'disabled' : '' ?> class="h-4 w-4 text-red-600 border-gray-300 rounded focus:ring-red-500">
                                        </td>
                                        
                                        <td class="border px-1 py-1">
                                            <input type="number" step="0.01" name="av4_<?= $aluno['id'] ?>" 
                                                value="<?= $av4 ?>" class="nota-av-input w-full border p-1 text-center" 
                                                data-aluno-id="<?= $aluno['id'] ?>" min="0" max="20" <?= $prazo_fechado ? 'readonly' : '' ?>>
                                        </td>
                                        <td class="border px-1 py-1 text-center">
                                            <input type="checkbox" name="falta_av4_<?= $aluno['id'] ?>" value="1" 
                                                <?= $falta_av4 == 1 ? 'checked' : '' ?> <?= $prazo_fechado ? 'disabled' : '' ?> class="h-4 w-4 text-red-600 border-gray-300 rounded focus:ring-red-500">
                                        </td>
                                        
                                        <td class="border px-1 py-1">
                                            <input type="number" step="0.01" name="av5_<?= $aluno['id'] ?>" 
                                                value="<?= $av5 ?>" class="nota-av-input w-full border p-1 text-center" 
                                                data-aluno-id="<?= $aluno['id'] ?>" min="0" max="20" <?= $prazo_fechado ? 'readonly' : '' ?>>
                                        </td>
                                        <td class="border px-1 py-1 text-center">
                                            <input type="checkbox" name="falta_av5_<?= $aluno['id'] ?>" value="1" 
                                                <?= $falta_av5 == 1 ? 'checked' : '' ?> <?= $prazo_fechado ? 'disabled' : '' ?> class="h-4 w-4 text-red-600 border-gray-300 rounded focus:ring-red-500">
                                        </td>

                                        <td class="border px-1 py-1">
                                            <input type="number" step="0.01" name="av6_<?= $aluno['id'] ?>" 
                                                value="<?= $av6 ?>" class="nota-av-input w-full border p-1 text-center" 
                                                data-aluno-id="<?= $aluno['id'] ?>" min="0" max="20" <?= $prazo_fechado ? 'readonly' : '' ?>>
                                        </td>
                                        <td class="border px-1 py-1 text-center">
                                            <input type="checkbox" name="falta_av6_<?= $aluno['id'] ?>" value="1" 
                                                <?= $falta_av6 == 1 ? 'checked' : '' ?> <?= $prazo_fechado ? 'disabled' : '' ?> class="h-4 w-4 text-red-600 border-gray-300 rounded focus:ring-red-500">
                                        </td>
                                        
                                        <td class="border px-2 py-1 text-center font-bold" id="mac_display_<?= $aluno['id'] ?>" data-nota-valor="<?= $mac ?>">
                                            <?= $mac !== '' ? number_format($mac, 2) : '' ?>
                                        </td>
                                        
                                        <input type="hidden" name="aluno_id[]" value="<?= $aluno['id'] ?>">
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>

                <div class="mt-6">
                    <?php if (!$prazo_fechado): ?>
                        <button type="submit" name="salvar_avs" class="bg-green-600 text-white px-6 py-3 rounded-lg hover:bg-green-700 transition"><i class="fas fa-save"></i> Salvar AVs, Faltas e MAC</button>
                    <?php else: ?>
                        <p class="text-red-600 font-semibold">⚠️ Prazo de lançamento fechado.</p>
                    <?php endif; ?>
                </div>
            </form>
        </div>
    <?php endif; ?>
</main>

<script>
    document.getElementById("menu-toggle").addEventListener("click", function() {
        document.getElementById("sidebar").classList.toggle("active");
        document.getElementById("overlay").classList.toggle("active");
    });
    document.getElementById("overlay").addEventListener("click", function() {
        document.getElementById("sidebar").classList.remove("active");
        this.classList.remove("active");
    });

    /**
     * Colore uma célula de nota se o valor for menor que 10.
     * @param {HTMLElement} element O elemento a ser colorido (célula ou input).
     * @param {number} valor O valor numérico da nota.
     */
    function colorirNota(element, valor) {
        // Remove classes de cor antes de aplicar a nova
        element.classList.remove('text-red-600', 'text-green-600'); 
        if (valor > 0 && valor < 10) {
            element.classList.add('text-red-600');
        } else if (valor >= 10) {
             element.classList.add('text-green-600');
        }
    }

    /**
     * Calcula e exibe a Média de Avaliações Contínuas (MAC) para um aluno em tempo real.
     * REGRA: Calcula MAC SÓ E SÓ SE o número de avaliações lançadas (> 0) for MAIOR OU IGUAL a 3.
     * (O cálculo NÃO é afetado pela marcação de falta.)
     * @param {string} alunoId O ID do aluno.
     */
    function calcularMAC(alunoId) {
        // Coleta todas as 6 AVs
        const avs = [
            parseFloat(document.querySelector(`input[name="av1_${alunoId}"]`).value) || 0,
            parseFloat(document.querySelector(`input[name="av2_${alunoId}"]`).value) || 0,
            parseFloat(document.querySelector(`input[name="av3_${alunoId}"]`).value) || 0,
            parseFloat(document.querySelector(`input[name="av4_${alunoId}"]`).value) || 0,
            parseFloat(document.querySelector(`input[name="av5_${alunoId}"]`).value) || 0,
            parseFloat(document.querySelector(`input[name="av6_${alunoId}"]`).value) || 0
        ];
        
        const macCell = document.getElementById(`mac_display_${alunoId}`);
        let soma_avs = 0;
        let contagem_avs = 0;
        let mac = 0;

        // Calcula a soma e conta quantas AVs são válidas (> 0)
        avs.forEach(av => {
            if (av > 0) {
                soma_avs += av;
                contagem_avs++;
            }
        });

        // Aplica a regra: calcula o MAC se houver 3 ou mais avaliações válidas
        if (contagem_avs >= 3) { 
            mac = soma_avs / contagem_avs;
        }

        const macArredondado = Math.round(mac * 100) / 100; // Arredonda para 2 casas

        // Atualiza a célula de exibição
        macCell.textContent = macArredondado > 0 ? macArredondado.toFixed(2) : '';
        macCell.dataset.notaValor = macArredondado;
        colorirNota(macCell, macArredondado);
    }

    document.addEventListener('DOMContentLoaded', () => {
        const avInputs = document.querySelectorAll('.nota-av-input');
        
        // 1. Aplica cor inicial e adiciona listener a todos os campos de nota
        avInputs.forEach(input => {
            const valorInicial = parseFloat(input.value) || 0;
            colorirNota(input, valorInicial);

            // Adiciona listener somente se o campo NÃO for readonly
            if (!input.readOnly) {
                input.addEventListener('input', (event) => {
                    const alunoId = event.target.dataset.alunoId;
                    const valor = parseFloat(event.target.value) || 0;
                    
                    colorirNota(event.target, valor);
                    calcularMAC(alunoId); // Recalcula o MAC sempre que uma AV muda
                });
            }
        });
        
        // 2. Colore as notas de MAC ao carregar (para notas já salvas)
        document.querySelectorAll('[data-nota-valor]').forEach(cell => {
            const valor = parseFloat(cell.dataset.notaValor);
            if (!isNaN(valor)) {
                colorirNota(cell, valor);
            }
        });
    });
</script>

</body>
</html>