* Este arquivo contém a lógica em JavaScript para calcular as médias (MT, MFD, MEC, MF)
 * e determinar o estado do aluno em tempo real, com base nas notas do MAC, NPT, NEO e NEE.
 *
 * Variáveis globais esperadas do PHP:
 * - todasAsNotas: Objeto JSON com todas as notas salvas, por aluno e trimestre.
 * - trimestreAtual: O número do trimestre atualmente selecionado (1, 2 ou 3).
 */

// ====================================================================
// 1. FUNÇÕES AUXILIARES
// ====================================================================

/**
 * Arredonda um valor para duas casas decimais (utilizado nas médias).
 * @param {number} valor
 * @returns {number} Valor arredondado.
 */
function arredondar(valor) {
    if (isNaN(valor) || !isFinite(valor)) {
        return null; // Retorna nulo se o valor for inválido
    }
    // Arredonda para 2 casas decimais
    return Math.round(valor * 100) / 100;
}

/**
 * Determina a classe CSS (aprovada/reprovada) com base na nota.
 * @param {number} nota
 * @returns {string} Classe CSS.
 */
function determinaClasse(nota) {
    if (nota === null || nota === undefined) return '';
    return (nota >= 10.0) ? 'nota-aprovada' : 'nota-reprovada';
}

/**
 * Determina o estado final do aluno (Aprovado, Reprovado, Incompleto).
 * @param {number} mf - Média Final
 * @returns {{texto: string, classe: string}}
 */
function determinaEstado(mf) {
    if (mf === null || mf === undefined) {
        return { texto: 'INCOMPLETO', classe: 'estado-final-incompleto' };
    }
    const mf_arredondada = arredondar(mf);
    if (mf_arredondada >= 10.0) {
        return { texto: 'APROVADO', classe: 'estado-final-aprovado' };
    } else {
        return { texto: 'REPROVADO', classe: 'estado-final-reprovado' };
    }
}


// ====================================================================
// 2. FUNÇÕES DE CÁLCULO
// ====================================================================

/**
 * Calcula a Média Trimestral (MT = (MAC + NPT) / 2).
 * @param {number} mac - Média de Avaliação Contínua
 * @param {number} npt - Nota de Prova Trimestral
 * @returns {number|null} MT arredondada ou null se faltar uma nota.
 */
function calculaMT(mac, npt) {
    if (mac === null || npt === null || mac < 0 || npt < 0) {
        return null;
    }
    return arredondar((mac + npt) / 2);
}

/**
 * Calcula a Média de Frequência Disciplinar (MFD = (MT1 + MT2 + MT3) / 3).
 * Requer as MTs dos 3 trimestres.
 * @param {string} alunoId
 * @returns {number|null} MFD arredondada ou null se faltar um MT.
 */
function calculaMFD(alunoId) {
    const notasAluno = todasAsNotas[alunoId] || {};
    const mt1 = parseFloat(notasAluno[1]?.mt);
    const mt2 = parseFloat(notasAluno[2]?.mt);
    const mt3 = parseFloat(notasAluno[3]?.mt);

    if (isNaN(mt1) || isNaN(mt2) || isNaN(mt3)) {
        return null;
    }

    return arredondar((mt1 + mt2 + mt3) / 3);
}

/**
 * Calcula a Média de Exame e de Classe (MEC).
 * MEC = (MFD + NEO/NEE) / 2
 * Assume-se que NEO e NEE não são preenchidos ao mesmo tempo.
 * @param {number} mfd - Média de Frequência Disciplinar
 * @param {number} neo - Nota de Exame Ordinário
 * @param {number} nee - Nota de Exame Especial
 * @returns {number|null} MEC arredondada ou null se faltar uma nota.
 */
function calculaMEC(mfd, neo, nee) {
    if (mfd === null) return null;

    let notaExame = null;
    if (neo !== null && !isNaN(neo) && neo >= 0) {
        notaExame = neo;
    } else if (nee !== null && !isNaN(nee) && nee >= 0) {
        notaExame = nee;
    }

    if (notaExame === null) return null; // Exame não lançado

    return arredondar((mfd + notaExame) / 2);
}

/**
 * Calcula a Média Final (MF).
 * Regra básica: Se MEC existe, MF é MEC. Caso contrário, se MFD >= 10, MF é MFD.
 * Se MFD < 10, MF será a MFD, mas o aluno estará reprovado.
 * @param {number} mfd
 * @param {number} mec
 * @returns {number|null} MF arredondada.
 */
function calculaMF(mfd, mec) {
    if (mec !== null) {
        return mec;
    }

    if (mfd !== null) {
        // Se não houve exame, MF é a MFD.
        return mfd;
    }

    return null;
}


// ====================================================================
// 3. FUNÇÃO PRINCIPAL DE ATUALIZAÇÃO DA LINHA
// ====================================================================

/**
 * Atualiza todos os campos calculados na linha da tabela de um aluno.
 * @param {string} alunoId
 * @param {HTMLElement} rowElement - O elemento <tr> do aluno
 */
function atualizaLinha(alunoId, rowElement) {
    const macCell = rowElement.querySelector('.mac-cell');
    const nptInput = rowElement.querySelector('.npt-input');
    const mtCell = rowElement.querySelector('.mt-cell');
    
    // 1. Obter MAC e NPT
    // MAC é lido do atributo data (pois é calculado na Caderneta, não editável aqui)
    const mac = parseFloat(macCell.dataset.macValor);
    const npt = parseFloat(nptInput.value);
    
    // 2. Calcular MT (Média Trimestral)
    const mt = calculaMT(mac, npt);
    
    // 3. Atualizar MT na célula e no campo hidden
    if (mt !== null) {
        const mt_formatado = mt.toFixed(2);
        mtCell.querySelector('.mt-valor').textContent = mt_formatado;
        rowElement.querySelector('.mt-input').value = mt_formatado;
        mtCell.dataset.notaValor = mt_formatado;
        
        // Atualiza a classe de cor
        mtCell.className = 'nota-calculada mt-cell print-value ' + determinaClasse(mt);
        
        // Salva a MT atualizada no objeto global para ser usada no MFD
        if (!todasAsNotas[alunoId]) todasAsNotas[alunoId] = {};
        todasAsNotas[alunoId][trimestreAtual] = { ...todasAsNotas[alunoId][trimestreAtual], mt: mt_formatado };
        
    } else {
        mtCell.querySelector('.mt-valor').textContent = '';
        rowElement.querySelector('.mt-input').value = '';
        mtCell.dataset.notaValor = '';
        mtCell.className = 'nota-calculada mt-cell print-value';
        
        // Remove MT do objeto global se estiver incompleto
        if (todasAsNotas[alunoId] && todasAsNotas[alunoId][trimestreAtual]) {
             delete todasAsNotas[alunoId][trimestreAtual].mt;
        }
    }
    
    // Aplica classe de cor ao NPT
    nptInput.parentElement.className = 'print-value ' + determinaClasse(npt);
    nptInput.parentElement.dataset.notaValor = nptInput.value;

    
    // 4. Cálculos Finais (Apenas se for o 3º Trimestre)
    if (trimestreAtual === 3) {
        const mfdCell = rowElement.querySelector('.mfd-cell');
        const neoInput = rowElement.querySelector('.neo-input');
        const neeInput = rowElement.querySelector('.nee-input');
        const mecCell = rowElement.querySelector('.mec-cell');
        const mfCell = rowElement.querySelector('.mf-cell');
        const estadoCell = rowElement.querySelector('.estado-cell');

        // A. MFD (Média de Frequência Disciplinar)
        const mfd = calculaMFD(alunoId);

        if (mfd !== null) {
            const mfd_formatado = mfd.toFixed(2);
            mfdCell.querySelector('.mfd-valor').textContent = mfd_formatado;
            rowElement.querySelector('.mfd-input').value = mfd_formatado;
            mfdCell.dataset.notaValor = mfd_formatado;
            mfdCell.className = 'nota-calculada mfd-cell print-value ' + determinaClasse(mfd);
        } else {
            mfdCell.querySelector('.mfd-valor').textContent = '';
            rowElement.querySelector('.mfd-input').value = '';
            mfdCell.dataset.notaValor = '';
            mfdCell.className = 'nota-calculada mfd-cell print-value';
        }

        // B. NEO/NEE (Notas de Exame)
        const neo = parseFloat(neoInput.value);
        const nee = parseFloat(neeInput.value);
        
        // Aplica classe de cor ao NEO e NEE
        neoInput.parentElement.className = 'print-value ' + determinaClasse(neo);
        neoInput.parentElement.dataset.notaValor = neoInput.value;
        neeInput.parentElement.className = 'print-value ' + determinaClasse(nee);
        neeInput.parentElement.dataset.notaValor = neeInput.value;


        // C. MEC (Média de Exame e de Classe)
        const mec = calculaMEC(mfd, neo, nee);

        if (mec !== null) {
            const mec_formatado = mec.toFixed(2);
            mecCell.querySelector('.mec-valor').textContent = mec_formatado;
            rowElement.querySelector('.mec-input').value = mec_formatado;
            mecCell.dataset.notaValor = mec_formatado;
            mecCell.className = 'nota-calculada mec-cell print-value ' + determinaClasse(mec);
        } else {
            mecCell.querySelector('.mec-valor').textContent = '';
            rowElement.querySelector('.mec-input').value = '';
            mecCell.dataset.notaValor = '';
            mecCell.className = 'nota-calculada mec-cell print-value';
        }

        // D. MF (Média Final)
        const mf = calculaMF(mfd, mec);
        const estado = determinaEstado(mf);

        if (mf !== null) {
            const mf_formatado = mf.toFixed(2);
            mfCell.querySelector('.mf-valor').textContent = mf_formatado;
            rowElement.querySelector('.mf-input').value = mf_formatado;
            mfCell.dataset.notaValor = mf_formatado;
            mfCell.className = 'nota-final-calc mf-cell print-value ' + determinaClasse(mf);
        } else {
            mfCell.querySelector('.mf-valor').textContent = '';
            rowElement.querySelector('.mf-input').value = '';
            mfCell.dataset.notaValor = '';
            mfCell.className = 'nota-final-calc mf-cell print-value';
        }

        // E. ESTADO FINAL
        estadoCell.querySelector('.estado-final-valor').textContent = estado.texto;
        estadoCell.dataset.notaValor = estado.texto;
        estadoCell.className = 'estado-cell print-value ' + estado.classe;
    }
}


// ====================================================================
// 4. INICIALIZAÇÃO E LISTENERS
// ====================================================================

function init() {
    // 1. Se não houver turma/disciplina/trimestre selecionados, sai
    if (trimestreAtual === 0) return;

    // 2. Seleciona todos os inputs de notas que disparam cálculo
    const inputsNotas = document.querySelectorAll('.npt-input, .neo-input, .nee-input');

    inputsNotas.forEach(input => {
        const row = input.closest('tr');
        const alunoId = row.dataset.alunoId;

        // Limita o valor máximo (20)
        input.addEventListener('input', (e) => {
            let value = parseFloat(e.target.value);
            if (value > 20) {
                e.target.value = 20;
            } else if (value < 0) {
                e.target.value = 0;
            } else if (isNaN(value)) {
                e.target.value = '';
            }
            atualizaLinha(alunoId, row);
        });

        // Dispara o cálculo inicial para preencher as células MT/MFD/MF/MEC/ESTADO
        atualizaLinha(alunoId, row);
    });
}

// Inicia o script quando a página estiver carregada
window.addEventListener('load', init);