<?php
session_start();
// Certifique-se de que a conexão está estabelecida corretamente
// MANTIDO: Assume-se que 'conexao.php' faz a conexão $conn com tratamento de erros.
include 'conexao.php'; 

// Variáveis para mensagens e controle de estado
$mensagem = '';
// MELHORIA 1: Usa array para capturar IDs da URL e limpar a URL de parâmetros temporários
$turma_id_url = $_GET['turma_id'] ?? null;
$aluno_id_editado_url = $_GET['aluno_id_editado'] ?? null;

// Remove os parâmetros temporários para ter uma URL base limpa
// Isso evita que o aluno editado seja carregado novamente após um simples F5
if (isset($_GET['aluno_id_editado'])) {
    // Redireciona para a mesma página sem o parâmetro aluno_id_editado
    $base_url = 'editar_aluno.php';
    $query_params = [];
    if ($turma_id_url) {
        $query_params['turma_id'] = $turma_id_url;
    }
    
    // Constrói a URL sem o aluno_id_editado
    $redirect_url = $base_url . (count($query_params) > 0 ? '?' . http_build_query($query_params) : '');
    
    // MELHORIA 2: Adiciona cabeçalho de cache para evitar problemas de re-submissão
    header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
    header("Pragma: no-cache");
    header("Expires: Sat, 26 Jul 1997 05:00:00 GMT"); 
    header("Location: " . $redirect_url);
    exit();
}

// --- TRATAMENTO DO FORMULÁRIO DE EDIÇÃO (POST) ---

if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['salvar_edicao'])) {
    
    // MELHORIA 3: Validação de ID principal
    $aluno_id = filter_input(INPUT_POST, 'aluno_id', FILTER_VALIDATE_INT);
    if (!$aluno_id) {
        $mensagem = "⚠️ Erro: Nenhum aluno válido selecionado para edição.";
        // Tenta capturar o ID da turma para redirecionamento, mesmo com erro
        $turma_id_url = $_POST['turma_id_edicao'] ?? $turma_id_url;
    } else {

        // Coleta dos dados e SANEAMENTO (limpeza)
        $nome_completo = trim(filter_input(INPUT_POST, 'nome_completo', FILTER_SANITIZE_SPECIAL_CHARS) ?? '');
        // MELHORIA 4: Usa filter_input para garantir que 'idade' e 'vez' são integers
        $idade = filter_input(INPUT_POST, 'idade', FILTER_VALIDATE_INT);
        $genero = filter_input(INPUT_POST, 'genero', FILTER_SANITIZE_SPECIAL_CHARS) ?? '';
        $data_nascimento = filter_input(INPUT_POST, 'data_nascimento', FILTER_SANITIZE_SPECIAL_CHARS) ?? null; // Data pode ser null/vazia
        $contacto = filter_input(INPUT_POST, 'contacto', FILTER_SANITIZE_SPECIAL_CHARS) ?? '';
        $endereco = filter_input(INPUT_POST, 'endereco', FILTER_SANITIZE_SPECIAL_CHARS) ?? '';
        
        $opcao_lingua = filter_input(INPUT_POST, 'opcao_lingua', FILTER_SANITIZE_SPECIAL_CHARS) ?? ''; 
        $vez = filter_input(INPUT_POST, 'vez', FILTER_VALIDATE_INT) ?? 1; // Garante um valor padrão
        $observacoes = filter_input(INPUT_POST, 'observacoes', FILTER_SANITIZE_SPECIAL_CHARS) ?? '';
        $foto_nome_atual = filter_input(INPUT_POST, 'foto_existente', FILTER_SANITIZE_SPECIAL_CHARS) ?? null;
        
        // NOVO: Coleta o ID da turma ativa para redirecionamento
        $turma_id_ativa = filter_input(INPUT_POST, 'turma_id_edicao', FILTER_VALIDATE_INT); 
        
        // Campos adicionais
        $numero_processo = filter_input(INPUT_POST, 'numero_processo', FILTER_SANITIZE_SPECIAL_CHARS) ?? '';
        $naturalidade = filter_input(INPUT_POST, 'naturalidade', FILTER_SANITIZE_SPECIAL_CHARS) ?? '';
        $provincia = filter_input(INPUT_POST, 'provincia', FILTER_SANITIZE_SPECIAL_CHARS) ?? '';
        $bi_numero = filter_input(INPUT_POST, 'bi_numero', FILTER_SANITIZE_SPECIAL_CHARS) ?? '';
        $bi_emitido_em = filter_input(INPUT_POST, 'bi_emitido_em', FILTER_SANITIZE_SPECIAL_CHARS) ?? '';
        $bi_data_emissao = filter_input(INPUT_POST, 'bi_data_emissao', FILTER_SANITIZE_SPECIAL_CHARS) ?? null;
        $municipio = filter_input(INPUT_POST, 'municipio', FILTER_SANITIZE_SPECIAL_CHARS) ?? '';
        $nome_pai = filter_input(INPUT_POST, 'nome_pai', FILTER_SANITIZE_SPECIAL_CHARS) ?? '';
        $nome_mae = filter_input(INPUT_POST, 'nome_mae', FILTER_SANITIZE_SPECIAL_CHARS) ?? '';


        // Lógica de upload de foto
        if (isset($_FILES['foto']) && $_FILES['foto']['error'] === UPLOAD_ERR_OK) {
            $ext = pathinfo($_FILES['foto']['name'], PATHINFO_EXTENSION);
            // MELHORIA 5: Adiciona um hash mais robusto e garante que o nome do arquivo seja limpo
            $foto_nome_candidata = uniqid('foto_') . hash('crc32', microtime()) . '.' . $ext; 

            // MELHORIA 6: Testa se a pasta existe ANTES de mover o arquivo
            $pasta_destino = "fotos_alunos/";
            if (!is_dir($pasta_destino)) {
                mkdir($pasta_destino, 0755, true);
            }

            if (move_uploaded_file($_FILES['foto']['tmp_name'], $pasta_destino . $foto_nome_candidata)) {
                 // Se o upload for bem-sucedido, substitui o nome da foto
                $foto_nome_atual = $foto_nome_candidata;
            } else {
                $mensagem = "❌ Erro ao mover o arquivo de foto. Verifique permissões da pasta: " . $pasta_destino;
                // Mantém a foto antiga em caso de erro no upload (foto_nome_atual já tem o valor anterior)
            }
        }
        
        // --- PREPARAÇÃO DO BIND_PARAM ---
        // Garante que $idade seja um número (0 se falhou no filtro ou não foi definido, ou seu valor se foi definido)
        $idade_bind = $idade === false || $idade === null ? 0 : $idade; 
        $vez_bind = $vez === false || $vez === null ? 1 : $vez; 

        // MELHORIA 7: Ajusta a string de tipos para refletir $idade e $vez como integers ('i')
        // Tipagem: s (nome), i (idade), s (genero), s (data_nasc), s (contacto), s (endereco), 
        // s (foto), s (opcao_lingua), i (vez), s (observacoes), s (naturalidade), s (provincia), 
        // s (bi_num), s (bi_emit_em), s (bi_data_emissao), s (municipio), s (nome_pai), 
        // s (nome_mae), s (num_proc), i (aluno_id) -> Total: s(17), i(3) = "sissssisississsssssi"
        $tipos = "sississsissssssssssi"; // String de tipos corrigida

        // Prepara a query de atualização (usando placeholders ? para segurança - Prepared Statements)
        $stmt = $conn->prepare("UPDATE alunos SET 
            nome = ?, idade = ?, genero = ?, data_nascimento = ?, contacto = ?, 
            endereco = ?, foto = ?, opcao_lingua = ?, vez = ?, observacoes = ?, 
            naturalidade = ?, provincia = ?, bi_numero = ?, bi_emitido_em = ?, 
            bi_data_emissao = ?, municipio = ?, nome_pai = ?, nome_mae = ?,
            numero_processo = ? 
            WHERE id = ?");
        
        if ($stmt === false) {
            // MELHORIA 8: Adiciona o erro do MySQL à mensagem para debugging
            $mensagem = "❌ Erro na preparação da query: " . $conn->error;
        } else {
            
            // MELHORIA 9: Chama bind_param com a string de tipos correta e as variáveis
            $bind_result = $stmt->bind_param($tipos, 
                $nome_completo, $idade_bind, $genero, $data_nascimento, $contacto, $endereco, 
                $foto_nome_atual, $opcao_lingua, $vez_bind, $observacoes, 
                $naturalidade, $provincia, $bi_numero, $bi_emitido_em, 
                $bi_data_emissao, $municipio, $nome_pai, $nome_mae, 
                $numero_processo, $aluno_id);
            
            if ($bind_result === false) {
                // MELHORIA 10: Tratamento de erro específico para o bind
                 $mensagem = "❌ Erro ao vincular parâmetros (bind_param): " . $stmt->error;
            } else if ($stmt->execute()) {
                $mensagem = "✅ Dados do aluno atualizados com sucesso!";
                
                // AJUSTE CRÍTICO: Redireciona para evitar re-submissão do formulário no F5
                if ($turma_id_ativa) {
                    // Adiciona o aluno_id_editado para que o JS possa selecioná-lo e carregar seus dados
                    header("Location: editar_aluno.php?turma_id=" . $turma_id_ativa . "&aluno_id_editado=" . $aluno_id);
                } else {
                    // Se não houver turma, redireciona apenas para a página base
                    header("Location: editar_aluno.php?aluno_id_editado=" . $aluno_id);
                }
                exit(); 
            } else {
                // MELHORIA 11: Tratamento de erro de execução
                $mensagem = "❌ Erro ao atualizar dados do aluno: " . $stmt->error;
            }
            $stmt->close();
        }
    }
}
// MELHORIA 12: Fecha a conexão com o banco de dados no final do script PHP
if (isset($conn)) {
    $conn->close();
}

// O restante do HTML e JS é mantido, mas com ajustes no JS (ver abaixo)

?>

<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <title>Editar Aluno</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    
    <style>
        /* Estilos mantidos para layout */
        * { box-sizing: border-box; }
        body { margin: 0; font-family: 'Segoe UI', sans-serif; background-color: #f0f2f5; }
        .topbar { background-color: #007bff; color: white; padding: 15px 20px; font-size: 20px; position: fixed; top: 0; left: 0; right: 0; z-index: 1000; }
        .sidebar { position: fixed; top: 60px; left: 0; width: 220px; height: 100%; background-color: #111827; padding-top: 20px; }
        .sidebar a { display: block; color: white; padding: 12px 20px; text-decoration: none; font-weight: 500; }
        .sidebar a:hover { background-color: #1e3a8a; }
        .main { margin-left: 220px; padding: 80px 30px 30px 30px; }
        .container { max-width: 1200px; background: #fff; padding: 30px; border-radius: 12px; box-shadow: 0 8px 20px rgba(0,0,0,0.1); }
        h2 { text-align: center; color: #007bff; margin-bottom: 25px; }
        .content-wrapper { display: flex; gap: 20px; }
        .list-container { flex: 1; min-width: 300px; max-height: 700px; overflow-y: auto; padding: 10px; border: 1px solid #ddd; border-radius: 8px; background-color: #f8f9fa; }
        .form-container { flex: 2; }
        .list-title { font-size: 1.2em; font-weight: 700; color: #343a40; margin-bottom: 10px; border-bottom: 2px solid #007bff; padding-bottom: 5px; }
        .list-item { padding: 10px; border-bottom: 1px solid #eee; cursor: pointer; transition: background-color 0.2s; }
        .list-item:hover { background-color: #e9ecef; }
        .list-item.active { background-color: #007bff; color: white; font-weight: 600; }
        .aluno-item { padding-left: 20px; font-size: 0.9em; }
        form { display: grid; grid-template-columns: 1fr 1fr; gap: 20px; }
        .full { grid-column: 1 / 3; }
        label { display: block; font-weight: 600; margin-bottom: 5px; }
        input, select, textarea { width: 100%; padding: 10px; border: 1px solid #ccc; border-radius: 8px; font-size: 14px; }
        input[type="file"] { padding: 6px; }
        button { background-color: #007bff; border: none; color: white; padding: 12px 20px; font-size: 16px; border-radius: 8px; cursor: pointer; transition: 0.3s; }
        button:hover { background-color: #0056b3; }
        .mensagem { margin-top: 20px; text-align: center; font-size: 16px; }
        .button-group { display: flex; justify-content: center; gap: 10px; }
        .info-box { background-color: #f9f9f9; padding: 10px; border: 1px dashed #ccc; border-radius: 8px; margin-top: 5px; }
        @media (max-width: 992px) {
            .content-wrapper { flex-direction: column; }
            .list-container { max-height: 300px; }
        }
        @media (max-width: 768px) {
            .sidebar { width: 100%; height: auto; position: relative; }
            .main { margin-left: 0; padding-top: 120px; }
            form { grid-template-columns: 1fr; }
            .full { grid-column: 1 / 2; }
            .button-group { flex-direction: column; }
        }
    </style>
</head>
<body>

<div class="topbar">
    📚 EMIL FRICK - Sistema de Gestão Escolar
</div>

<div class="sidebar">
    <a href="painel_secretaria_geral.php">🏠 Início</a>
    <a href="matricular_aluno.php">📋 Matricular Novo</a>
    <a href="editar_aluno.php">✏️ Editar Alunos</a>
    <a href="cadastrar_turma1.php">🏫 Turmas</a>
    <a href="login.php">🔓 Sair</a>
</div>

<div class="main">
    <div class="container">
        <h2>✏️ Editar Dados do Aluno</h2>
        <div class="full" style="text-align: center;">
            <?php if (!empty($mensagem)): ?>
                <div class="mensagem">
                    <?= $mensagem ?>
                </div>
            <?php endif; ?>
        </div>
        
        <div class="content-wrapper">
            <div class="list-container">
                <div class="list-title">Selecione a Turma</div>
                <div id="lista_turmas">
                    Carregando turmas...
                </div>

                <div class="list-title" style="margin-top: 20px;">Alunos da Turma Selecionada</div>
                <div id="lista_alunos">
                    Nenhuma turma selecionada.
                </div>
            </div>

            <div class="form-container">
                <form method="POST" action="" enctype="multipart/form-data" id="formEdicao">
                    <input type="hidden" name="aluno_id" id="aluno_id" value="">
                    <input type="hidden" name="foto_existente" id="foto_existente" value="">
                    <input type="hidden" name="turma_id_edicao" id="turma_id_edicao" value=""> 
                    
                    <div class="full">
                        <label>Status:</label>
                        <div class="info-box" id="status_selecao">
                            Aguardando a seleção de um aluno.
                        </div>
                    </div>
                    
                    <div class="full">
                        <label>Nome Completo:</label>
                        <input type="text" name="nome_completo" id="nome_completo" value="" required disabled>
                    </div>
                    
                    <div class="full">
                        <label>Número de Processo:</label>
                        <input type="text" name="numero_processo" id="numero_processo" value="" disabled>
                    </div>
                    <div>
                        <label>Idade:</label>
                        <select name="idade" id="idade" required disabled>
                            <option value="">-- Selecione --</option>
                            <?php for ($i = 12; $i <= 60; $i++): ?>
                                <option value="<?= $i ?>"><?= $i ?> anos</option>
                            <?php endfor; ?>
                        </select>
                    </div>

                    <div>
                        <label>Género:</label>
                        <select name="genero" id="genero" required disabled>
                            <option value="">-- Selecione --</option>
                            <option value="Masculino">Masculino</option>
                            <option value="Feminino">Feminino</option>
                            <option value="Outro">Outro</option>
                        </select>
                    </div>

                    <div>
                        <label>Data de Nascimento:</label>
                        <input type="date" name="data_nascimento" id="data_nascimento" value="" disabled>
                    </div>
                    
                    <div class="full">
                        <label>Nº do Bilhete de Identidade (BI):</label>
                        <input type="text" name="bi_numero" id="bi_numero" value="" disabled>
                    </div>
                    <div>
                        <label>Local de Emissão do BI:</label>
                        <input type="text" name="bi_emitido_em" id="bi_emitido_em" value="" disabled>
                    </div>
                    <div>
                        <label>Data de Emissão do BI:</label>
                        <input type="date" name="bi_data_emissao" id="bi_data_emissao" value="" disabled>
                    </div>

                    <div>
                        <label>Naturalidade:</label>
                        <input type="text" name="naturalidade" id="naturalidade" value="" disabled>
                    </div>

                    <div>
                        <label>Província:</label>
                        <input type="text" name="provincia" id="provincia" value="" disabled>
                    </div>

                    <div>
                        <label>Município:</label>
                        <input type="text" name="municipio" id="municipio" value="" disabled>
                    </div>

                    <div class="full">
                        <label>Nome do Pai:</label>
                        <input type="text" name="nome_pai" id="nome_pai" value="" disabled>
                    </div>

                    <div class="full">
                        <label>Nome da Mãe:</label>
                        <input type="text" name="nome_mae" id="nome_mae" value="" disabled>
                    </div>

                    <div>
                        <label>Contacto:</label>
                        <input type="text" name="contacto" id="contacto" value="" disabled>
                    </div>

                    <div>
                        <label>Endereço:</label>
                        <input type="text" name="endereco" id="endereco" value="" disabled>
                    </div>
                    
                    <div class="full" id="turma_selecao_container">
                        <label>Turma Atual (Classe, Nome, Turno, Ano):</label>
                        <div class="info-box" id="turma_display">N/A</div>
                    </div>
                    
                    <div class="full">
                        <label>Foto do Aluno (Substituir):</label>
                        <input type="file" name="foto" disabled id="foto_input">
                        
                        <div style="margin-top: 10px; text-align: center;">
                            <img id="foto_atual_display" src="" alt="Foto do Aluno Atual" 
                                style="max-width: 150px; max-height: 200px; border: 1px solid #ccc; display: none;">
                            <p id="foto_status">Nenhuma foto carregada.</p>
                        </div>
                    </div>

                    <div>
                        <label>Opção de Língua:</label>
                        <select name="opcao_lingua" id="opcao_lingua" disabled>
                            <option value="">-- Selecione (Opcional) --</option>
                            <option value="Português">Português</option>
                            <option value="Inglês">Inglês</option>
                            <option value="Francês">Francês</option>
                        </select>
                    </div>

                    <div>
                        <label>Vez da Matrícula:</label>
                        <select name="vez" id="vez" required disabled>
                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                <option value="<?= $i ?>"><?= $i ?>ª Vez</option>
                            <?php endfor; ?>
                        </select>
                    </div>
                    
                    <div class="full">
                        <label>Observações:</label>
                        <textarea name="observacoes" id="observacoes" rows="4" disabled></textarea>
                    </div>

                    <div class="full" style="text-align:center;">
                        <div class="button-group">
                            <button type="submit" name="salvar_edicao" id="btn_salvar_edicao" disabled>✅ Salvar Alterações</button>
                        </div>
                    </div>
                </form>
            </div>
        </div> </div>
</div>

<script>
// MELHORIA 13: Ajustes no JavaScript para lidar com o estado da URL e evitar re-seleção desnecessária
$(function() {
    
    // Lista de TODOS os campos que serão habilitados/desabilitados
    const camposFormulario = [
        '#nome_completo', '#numero_processo', '#idade', '#genero', '#data_nascimento', 
        '#contacto', '#endereco', '#opcao_lingua', '#vez', '#observacoes', 
        '#foto_input', '#btn_salvar_edicao',
        '#bi_numero', '#bi_emitido_em', '#bi_data_emissao', 
        '#naturalidade', '#provincia', '#municipio', 
        '#nome_pai', '#nome_mae'
    ];
    
    // Variáveis de Estado
    let turmaAtivaId = null; 
    
    // Captura os IDs da URL
    const urlParams = new URLSearchParams(window.location.search);
    const turmaIdNaURL = urlParams.get('turma_id');
    const alunoIdNaURL = urlParams.get('aluno_id_editado');

    // Remove o parâmetro temporário do aluno, se houver, para a próxima interação
    // O PHP já deve ter redirecionado, mas essa checagem pode ser útil
    // No PHP, já removemos o alunoIdNaURL com o redirecionamento.
    
    // 1. Função para habilitar/desabilitar o formulário
    function toggleFormulario(habilitar) {
        camposFormulario.forEach(selector => {
            $(selector).prop('disabled', !habilitar);
        });
        $('#status_selecao').text(habilitar ? 'Dados carregados e prontos para edição.' : 'Aguardando a seleção de um aluno.');
    }

    // 2. Função para limpar e bloquear o formulário
    function limparEBloquear(mensagemTurma) {
        // Zera os valores chave
        $("#aluno_id").val("");
        // Reseta todos os campos do formulário para os valores padrão
        $("#formEdicao")[0].reset(); 
        
        // Atualiza display
        $('#turma_display').text(mensagemTurma || "N/A");
        $('#turma_id_edicao').val(turmaAtivaId); // Mantém o ID da turma ativa no hidden field
        
        // Esconde e reseta a foto
        $('#foto_atual_display').attr('src', '').hide();
        $('#foto_status').text("Nenhuma foto carregada.");
        
        // Remove a seleção visual do aluno ativo
        $('#lista_alunos .aluno-item.active').removeClass('active');
        
        toggleFormulario(false);
    }
    
    // 3. Função para preencher os campos do formulário (MANTIDA)
    function preencherDadosAluno(dados) {
        $("#aluno_id").val(dados.id);
        $("#foto_existente").val(dados.foto); 
        
        // Preenchimento dos dados (tratando null/undefined para strings vazias)
        $("#nome_completo").val(dados.nome || '');
        $("#numero_processo").val(dados.numero_processo || ''); 
        $("#data_nascimento").val(dados.data_nascimento || '');
        $("#contacto").val(dados.contacto || '');
        $("#endereco").val(dados.endereco || '');
        $("#bi_numero").val(dados.bi_numero || '');
        $("#bi_emitido_em").val(dados.bi_emitido_em || '');
        $("#bi_data_emissao").val(dados.bi_data_emissao || '');
        $("#naturalidade").val(dados.naturalidade || '');
        $("#provincia").val(dados.provincia || '');
        $("#municipio").val(dados.municipio || '');
        $("#nome_pai").val(dados.nome_pai || '');
        $("#nome_mae").val(dados.nome_mae || '');
        $("#observacoes").val(dados.observacoes || ''); 
        
        // Selects
        $("#idade").val(dados.idade);
        $("#genero").val(dados.genero);
        $("#opcao_lingua").val(dados.opcao_lingua || ''); 
        $("#vez").val(dados.vez);
        
        // Informação da Turma (Exibição)
        const nomeTurmaDisplay = dados.nome_turma_display || 'N/A';
        const turmaTexto = 
            `${dados.classe ? dados.classe + 'ª Classe' : 'N/A'} - ` + 
            `${nomeTurmaDisplay} - ` +
            `${dados.curso || 'N/A'} (${dados.turno || 'N/A'})`;
            
        $('#turma_display').text(turmaTexto);
        $('#turma_id_edicao').val(dados.turma_id); 
        
        // Exibição da foto atual
        const fotoNome = dados.foto;
        const fotoDisplay = $('#foto_atual_display');
        const fotoStatus = $('#foto_status');

        if (fotoNome) {
            const caminhoCompleto = "fotos_alunos/" + fotoNome; 
            fotoDisplay.attr('src', caminhoCompleto).show();
            fotoStatus.text("Foto atual carregada.");
        } else {
            fotoDisplay.attr('src', '').hide();
            fotoStatus.text("Aluno não possui foto cadastrada.");
        }
        
        // HABILITA A EDIÇÃO
        toggleFormulario(true);
    }

    // 4.1. Carrega todas as Turmas
    function carregarTurmas() {
        $.ajax({
            url: "buscar_turmas.php", 
            type: "GET",
            dataType: "json",
            success: function(turmas) {
                let html = '';
                
                // Prioriza a turma da URL (apenas na inicialização)
                const turmaParaAtivar = turmaIdNaURL; 

                if (turmas.length > 0) {
                    turmas.forEach(function(turma) {
                        // Captura o nome completo para exibir no status
                        const nomeCompletoTurma = `${turma.classe ? turma.classe + 'ª Classe' : 'N/A'} - ${turma.nome || 'Turma S/ Nome'} (${turma.turno || 'S/ Turno'})`;
                        
                        // Verifica se deve ser ativada visualmente
                        const ativo = (turma.id == turmaParaAtivar) ? ' active' : ''; 

                        html += `<div class="list-item turma-item${ativo}" data-id="${turma.id}" data-nome-completo="${nomeCompletoTurma}">
                                        ${nomeCompletoTurma}
                                    </div>`;
                    });
                } else {
                    html = '<div class="info-box">Nenhuma turma cadastrada.</div>';
                }
                $('#lista_turmas').html(html);
                
                // AJUSTE: Se há uma turma para ativar (da URL), define o estado e carrega os alunos
                if (turmaParaAtivar) {
                    turmaAtivaId = turmaParaAtivar; 
                    limparEBloquear($(`.turma-item[data-id="${turmaParaAtivar}"]`).data('nome-completo'));
                    carregarAlunosPorTurma(turmaParaAtivar);
                }
            },
            error: function() {
                $('#lista_turmas').html('<div class="info-box" style="color: red;">Erro ao carregar turmas.</div>');
            }
        });
    }

    // 4.2. Carrega os Alunos de uma Turma
    function carregarAlunosPorTurma(turmaId) {
        $('#lista_alunos').html('Carregando alunos...');
        
        // Armazena a turma ativa para uso no POST
        turmaAtivaId = turmaId; 
        
        $.ajax({
            url: "buscar_alunos_por_turma.php", 
            type: "GET",
            dataType: "json",
            data: { turma_id: turmaId }, 
            success: function(alunos) {
                let html = '';
                
                if (alunos.length > 0) {
                    alunos.forEach(function(aluno) {
                        // Verifica se é o aluno editado (vindo da URL)
                        const ativo = (aluno.id == alunoIdNaURL) ? ' active' : ''; 
                        
                        html += `<div class="list-item aluno-item${ativo}" data-id="${aluno.id}">
                                        ${aluno.nome}
                                    </div>`;
                    });
                } else {
                    html = '<div class="info-box">Nenhum aluno matriculado nesta turma.</div>';
                }
                $('#lista_alunos').html(html);
                
                // AJUSTE CRÍTICO: Se houver um aluno editado na URL, carrega seus dados automaticamente
                if (alunoIdNaURL && $('.aluno-item.active').length > 0) {
                    // Após carregar, limpa a variável da URL para não recarregar no futuro
                    const idParaCarregar = alunoIdNaURL;
                    urlParams.delete('aluno_id_editado');
                    window.history.replaceState({}, document.title, "?" + urlParams.toString());
                    carregarDadosAluno(idParaCarregar);
                } 
            },
            error: function() {
                $('#lista_alunos').html('<div class="info-box" style="color: red;">Erro ao carregar alunos.</div>');
            }
        });
    }

    // 4.3. Carrega os Dados Completos de um Aluno (MANTIDA)
    function carregarDadosAluno(alunoId) {
        limparEBloquear("Carregando dados...");
        
        // Garante que o item da lista fique ativo ao carregar
        $('.aluno-item').removeClass('active');
        $(`.aluno-item[data-id="${alunoId}"]`).addClass('active');

        $.ajax({
            url: "buscar_aluno_dados.php", 
            type: "GET",
            dataType: "json",
            data: { aluno_id: alunoId }, 
            success: function(response) {
                if (response && response.length > 0) {
                    const dadosAluno = response[0];
                    preencherDadosAluno(dadosAluno);
                } else {
                    limparEBloquear("Dados não encontrados. Selecione novamente.");
                }
            },
            error: function(xhr, status, error) {
                console.error("Erro na comunicação AJAX:", status, error); 
                limparEBloquear("Erro de comunicação com o servidor.");
            }
        });
    }
    
    // 5. Listeners de Eventos (MANTIDOS)
    
    // Listener para o clique na Turma
    $('#lista_turmas').on('click', '.turma-item', function() {
        const turmaId = $(this).data('id');
        
        // Verifica se a turma clicada é a que já está ativa (para evitar recarregar)
        if (turmaAtivaId === turmaId) {
            return;
        }
        
        // Remove a classe ativa de todas as turmas
        $('.turma-item').removeClass('active');
        // Adiciona a classe ativa à turma clicada
        $(this).addClass('active');
        
        turmaAtivaId = turmaId; // Define a turma ativa

        // Limpa a seleção do aluno e o formulário
        limparEBloquear(`Turma Ativa: ${$(this).data('nome-completo')}`);
        
        // Carrega os alunos da turma selecionada
        carregarAlunosPorTurma(turmaId);
    });

    // Listener para o clique no Aluno
    $('#lista_alunos').on('click', '.aluno-item', function() {
        const alunoId = $(this).data('id');
        
        // Remove a classe ativa de todos os alunos
        $('.aluno-item').removeClass('active');
        // Adiciona a classe ativa ao aluno clicado
        $(this).addClass('active');

        // Carrega os dados do aluno e preenche o formulário
        carregarDadosAluno(alunoId);
    });

    // 6. Inicialização
    // 6.1. Inicializa o formulário e carrega a lista de turmas.
    limparEBloquear("Selecione uma turma e depois um aluno para iniciar a edição."); 
    
    // 6.2. Carrega a lista de turmas. A função fará a re-seleção se turmaIdNaURL estiver definido.
    carregarTurmas();
});
</script>

</body>
</html>