<?php
session_start();
include 'conexao.php';

// Redireciona se o usuário não estiver logado.
if (!isset($_SESSION['usuario'])) {
    header("Location: login.php");
    exit;
}

// O ID do remetente será o ID do aluno logado.
$remetente_id = $_SESSION['usuario']['id'];
$remetente_nome = $_SESSION['usuario']['nome'] ?? 'Aluno'; // Usar o nome do aluno logado

// Inicializa as variáveis para evitar a mensagem de "Undefined variable"
$sucesso = false;
$erro = '';

// Diretório onde os trabalhos serão salvos.
$diretorio_upload = 'uploads_trabalhos/';

// Enviar documento via POST.
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Valida se todos os campos estão preenchidos.
    if (isset($_POST['destinatario_id']) && isset($_POST['nome_grupo']) && isset($_POST['turma']) && isset($_FILES['documento'])) {
        $destinatario_id = intval($_POST['destinatario_id']);
        $nome_grupo = trim($_POST['nome_grupo']);
        $turma = trim($_POST['turma']);
        $arquivo = $_FILES['documento'];

        // Cria o título do documento combinando o nome do grupo/aluno e a turma.
        $titulo = "Trabalho de " . htmlspecialchars($nome_grupo) . " - Turma: " . htmlspecialchars($turma);

        // Verifica se houve sucesso no upload.
        if ($arquivo['error'] === 0 && $destinatario_id && !empty($nome_grupo) && !empty($turma)) {

            // --- Lógica de verificação e criação da pasta ---
            // Verifica se o diretório de uploads existe. Se não, tenta criá-lo.
            if (!is_dir($diretorio_upload)) {
                // Tenta criar o diretório com permissão 0777 (leitura, escrita e execução para todos).
                if (!mkdir($diretorio_upload, 0777, true)) {
                    $erro = "Falha ao criar o diretório de uploads. Verifique as permissões da pasta.";
                }
            }

            // Se o diretório existir ou foi criado com sucesso, continua o processo.
            if (empty($erro)) {
                // Cria um nome único para o arquivo usando o timestamp.
                $nome_arquivo = time() . '_' . basename($arquivo['name']);
                $caminho = $diretorio_upload . $nome_arquivo;

                // Move o arquivo temporário para a pasta de uploads.
                if (move_uploaded_file($arquivo['tmp_name'], $caminho)) {
                    // Insere os dados do documento no banco de dados.
                    $stmt = $conn->prepare("INSERT INTO documentos_enviados (remetente_id, destinatario_id, titulo, caminho_arquivo) VALUES (?, ?, ?, ?)");
                    $stmt->bind_param("iiss", $remetente_id, $destinatario_id, $titulo, $caminho);
                    
                    if ($stmt->execute()) {
                        $sucesso = true;
                    } else {
                        $erro = "Erro ao salvar no banco de dados: " . $stmt->error;
                    }
                    $stmt->close();
                } else {
                    $erro = "Erro ao mover o arquivo para o diretório de destino. Verifique as permissões.";
                }
            }
        } else {
            $erro = "Por favor, preencha todos os campos e envie um arquivo válido.";
        }
    } else {
        $erro = "Dados do formulário incompletos.";
    }
}

// Buscar apenas professores para a lista de destinatários.
// Assume-se que o aluno irá enviar o documento para um professor.
$res = $conn->query("SELECT id, nome FROM usuarios WHERE perfil = 'professor' ORDER BY nome ASC");
$professores = $res->fetch_all(MYSQLI_ASSOC);
$res->close();
?>

<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Enviar Documento</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        body {
            font-family: Arial, sans-serif;
            padding: 30px;
            background: #f5f5f5;
        }
        form {
            background: white;
            padding: 25px;
            max-width: 600px;
            margin: auto;
            border-radius: 8px;
            box-shadow: 0 0 10px #ccc;
        }
        h2 {
            text-align: center;
            color: #003366;
        }
        label {
            display: block;
            margin-top: 15px;
            font-weight: bold;
        }
        input, select {
            width: 100%;
            padding: 8px;
            margin-top: 5px;
            border: 1px solid #ccc;
            border-radius: 5px;
        }
        button {
            margin-top: 20px;
            background: #007bff;
            color: white;
            border: none;
            padding: 10px 15px;
            border-radius: 6px;
            cursor: pointer;
        }
        button:hover {
            background: #0056b3;
        }
        .mensagem {
            text-align: center;
            font-weight: bold;
            margin-top: 15px;
        }
        .sucesso {
            color: green;
        }
        .erro {
            color: red;
        }
    </style>
</head>
<body>

<h2>📤 Enviar Documento</h2>

<form method="POST" enctype="multipart/form-data">
    <label for="nome_grupo">Nome Completo ou Grupo:</label>
    <input type="text" name="nome_grupo" id="nome_grupo" placeholder="Ex: João da Silva ou Grupo Alfa" required>
    
    <label for="turma">Turma:</label>
    <input type="text" name="turma" id="turma" placeholder="Ex: 12ª Classe, Turma C" required>
    
    <label for="destinatario_id">Enviar para:</label>
    <select name="destinatario_id" required>
        <option value="">-- Selecionar Professor --</option>
        <?php foreach ($professores as $professor): ?>
            <option value="<?= $professor['id'] ?>"><?= htmlspecialchars($professor['nome']) ?></option>
        <?php endforeach; ?>
    </select>

    <label for="documento">Escolher Arquivo:</label>
    <input type="file" name="documento" accept=".pdf,.doc,.docx,.jpg,.png,.xlsx,.xls" required>

    <button type="submit">📤 Enviar Documento</button>

    <?php if ($sucesso): ?>
        <p class="mensagem sucesso">✅ Documento enviado com sucesso!</p>
    <?php elseif (!empty($erro)): ?>
        <p class="mensagem erro">❌ <?= $erro ?></p>
    <?php endif; ?>
</form>

</body>
</html>