<?php
session_start();
include 'conexao.php';

// Verifica se o usuário está logado e tem o perfil correto
$perfil = strtoupper($_SESSION['usuario']['perfil'] ?? '');
if (!in_array($perfil, ['SUBDIRECÇÃO PEDAGÓGICA', 'SECRETARIA PEDAGÓGICA'])) {
    header("Location: login.php");
    exit;
}

// Configuração de data e fuso
setlocale(LC_TIME, 'pt_PT.UTF-8', 'pt_BR.UTF-8', 'portuguese');
date_default_timezone_set('Africa/Luanda');

// Recebe parâmetros de forma segura
$turma_id = filter_input(INPUT_GET, 'turma_id', FILTER_VALIDATE_INT);
$trimestre = filter_input(INPUT_GET, 'trimestre', FILTER_VALIDATE_INT);

$nome_turma = $classe_turma = $curso_turma = $trimestre_extenso = '';
$ano_lectivo = date('Y');
$estatisticas_gerais = [
    'matriculados' => ['M' => 0, 'F' => 0, 'Total' => 0],
    'aprovados' => ['M' => 0, 'F' => 0, 'Total' => 0],
    'reprovados' => ['M' => 0, 'F' => 0, 'Total' => 0],
    'abandono' => ['M' => 0, 'F' => 0, 'Total' => 0],
    'transferido' => ['M' => 0, 'F' => 0, 'Total' => 0]
];

if ($trimestre == 1) {
    $trimestre_extenso = 'I Trimestre';
} elseif ($trimestre == 2) {
    $trimestre_extenso = 'II Trimestre';
} elseif ($trimestre == 3) {
    $trimestre_extenso = 'III Trimestre';
}

if ($turma_id && $trimestre) {
    // 1. Buscar dados da Turma
    $stmt_turma = $conn->prepare("SELECT nome, classe, curso, ano FROM turmas WHERE id = ?");
    $stmt_turma->bind_param("i", $turma_id);
    $stmt_turma->execute();
    $res_turma = $stmt_turma->get_result();
    if ($row_turma = $res_turma->fetch_assoc()) {
        $nome_turma = htmlspecialchars($row_turma['nome']);
        $classe_turma = htmlspecialchars($row_turma['classe']);
        $curso_turma = htmlspecialchars($row_turma['curso']);
        $ano_lectivo = $row_turma['ano'] ?? date('Y');
    }

    // 2. Buscar alunos da turma
    // Incluindo abandono e transferido - ASSUME QUE O SQL JÁ FOI EXECUTADO
    $stmt_alunos = $conn->prepare("SELECT id, nome, genero, abandono, transferido FROM alunos WHERE turma_id = ? ORDER BY nome");
    $stmt_alunos->bind_param("i", $turma_id);
    $stmt_alunos->execute();
    $alunos = $stmt_alunos->get_result()->fetch_all(MYSQLI_ASSOC);

    // 3. Buscar disciplinas da turma (para calcular a média por aluno)
    $stmt_disciplinas = $conn->prepare("SELECT id, nome FROM disciplinas WHERE turma_id = ?");
    $stmt_disciplinas->bind_param("i", $turma_id);
    $stmt_disciplinas->execute();
    $disciplinas = $stmt_disciplinas->get_result()->fetch_all(MYSQLI_ASSOC);
    $total_disciplinas = count($disciplinas);

    // Prepara statements para notas
    $stmt_mac = $conn->prepare("SELECT mac FROM avaliacoes_continuas WHERE aluno_id = ? AND disciplina_id = ? AND trimestre = ?");
    $stmt_notas = $conn->prepare("SELECT npp, npt FROM notas WHERE aluno_id = ? AND disciplina_id = ? AND trimestre = ?");

    // Loop principal de Cálculo da Média Global por Aluno
    foreach ($alunos as $aluno) {
        $genero = strtoupper(substr($aluno['genero'] ?? 'M', 0, 1));
        $genero = ($genero === 'F') ? 'F' : 'M';
        
        $estatisticas_gerais['matriculados'][$genero]++;
        $estatisticas_gerais['matriculados']['Total']++;
        
        $soma_medias_trimestrais = 0;
        $disciplinas_avaliadas = 0;

        // Verifica Abandono/Transferidos (Movimento da Turma)
        if (($aluno['abandono'] ?? 0) == 1) {
            $estatisticas_gerais['abandono'][$genero]++;
            $estatisticas_gerais['abandono']['Total']++;
            continue; // Pula o cálculo de aprovação para alunos que abandonaram
        }
        if (($aluno['transferido'] ?? 0) == 1) {
            $estatisticas_gerais['transferido'][$genero]++;
            $estatisticas_gerais['transferido']['Total']++;
            continue; // Pula o cálculo de aprovação para alunos que transferiram
        }
        
        // Loop por Disciplina para calcular a Média Global
        foreach ($disciplinas as $disc) {
            $disc_id = $disc['id'];
            $mac = 0; $npp = 0; $npt = 0;
            
            // Busca MAC
            $stmt_mac->bind_param("iii", $aluno['id'], $disc_id, $trimestre);
            $stmt_mac->execute();
            $res_mac = $stmt_mac->get_result()->fetch_assoc();
            $mac = floatval($res_mac['mac'] ?? 0);

            // Busca NPP e NPT
            $stmt_notas->bind_param("iii", $aluno['id'], $disc_id, $trimestre);
            $stmt_notas->execute();
            $res_notas = $stmt_notas->get_result()->fetch_assoc();
            $npp = floatval($res_notas['npp'] ?? 0);
            $npt = floatval($res_notas['npt'] ?? 0);
            
            if ($mac > 0 || $npp > 0 || $npt > 0) {
                $media_trimestral = ($mac + $npp + $npt) / 3;
                $soma_medias_trimestrais += $media_trimestral;
                $disciplinas_avaliadas++;
            }
        } // Fim Loop Disciplinas
        
        // Cálculo da Média Geral do Aluno no Trimestre
        if ($disciplinas_avaliadas > 0) {
            $media_geral_aluno = $soma_medias_trimestrais / $disciplinas_avaliadas;
            
            // Verifica Aprovação Global (Média Geral >= 10)
            if (round($media_geral_aluno) >= 10) {
                $estatisticas_gerais['aprovados'][$genero]++;
                $estatisticas_gerais['aprovados']['Total']++;
            } else {
                $estatisticas_gerais['reprovados'][$genero]++;
                $estatisticas_gerais['reprovados']['Total']++;
            }
        } else {
            // Se não foi avaliado em nada, é considerado Reprovado ou Pendente
            // Neste contexto, vamos classificá-lo como Reprovado por omissão de notas
             $estatisticas_gerais['reprovados'][$genero]++;
             $estatisticas_gerais['reprovados']['Total']++;
        }
    } // Fim Loop Alunos

    // Fecha statements
    $stmt_mac->close();
    $stmt_notas->close();
}

// Cálculo de percentagens finais
$matriculados_total_efetivo = $estatisticas_gerais['matriculados']['Total'] - $estatisticas_gerais['abandono']['Total'] - $estatisticas_gerais['transferido']['Total'];
$aprovados_total = $estatisticas_gerais['aprovados']['Total'];

$aproveitamento_geral = ($matriculados_total_efetivo > 0) ? 
    round(($aprovados_total / ($aprovados_total + $estatisticas_gerais['reprovados']['Total'])) * 100, 1) : 0;
$rendimento_geral = ($estatisticas_gerais['matriculados']['Total'] > 0) ? 
    round(($aprovados_total / $estatisticas_gerais['matriculados']['Total']) * 100, 1) : 0;

$data_atual = date('d/m/Y');
?>

<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Estatística Global por Turma</title>
    <style>
        body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; margin: 0; background-color: #f4f7f9; color: #333; display: flex; justify-content: center; flex-direction: column; align-items: center; padding: 10px; box-sizing: border-box; }
        .container { max-width: 800px; width: 95%; background-color: #ffffff; padding: 25px; box-shadow: 0 2px 10px rgba(0, 0, 0, 0.15); border-radius: 8px; position: relative; }
        .cabecalho { text-align: center; margin-bottom: 25px; line-height: 1.1; border-bottom: 2px solid #003366; padding-bottom: 10px; }
        h2, h3 { margin: 0; font-weight: 700; font-size: 1.4em; color: #003366; }
        h3 { font-size: 1.1em; font-weight: 500; color: #555; }
        .info-trimestre-turma { text-align: center; margin-bottom: 20px; font-size: 1.1em; font-weight: bold; padding: 5px 0; background-color: #e6f0ff; border-radius: 4px; }
        table { border-collapse: collapse; width: 100%; margin-top: 20px; font-size: 14px; }
        th, td { border: 1px solid #ccc; padding: 10px; text-align: center; }
        thead th { background-color: #003366; color: white; font-weight: 600; }
        .col-total { font-weight: bold; background-color: #f1f1f1; }
        .resultado { font-size: 1.2em; font-weight: bold; color: #003366; }
        .percentagem { font-size: 1.3em; font-weight: 700; color: #28a745; background-color: #e6ffe6; }
        
        .assinaturas { display: flex; justify-content: space-between; margin-top: 50px; font-size: 0.9em; }
        .assinatura-esquerda, .assinatura-direita { width: 45%; text-align: center; }
        .line { border-bottom: 1px solid #333; margin-top: 60px; }
        .data-mapa { text-align: right; margin-top: 30px; font-size: 0.9em; }
        .btn-back { background-color: #6c757d; color: white; padding: 8px 15px; border-radius: 5px; text-decoration: none; margin-bottom: 15px; display: inline-block; }

        @media print {
            body { background-color: #fff; padding: 0; }
            .btn-back { display: none; }
            .container { box-shadow: none; padding: 0; max-width: 100%; font-size: 12px; }
            table { font-size: 11px; }
            th, td { padding: 5px; }
            .cabecalho { border-bottom: 1px solid #ccc; }
        }
    </style>
</head>
<body>

<a href="estatistica_turma.php?turma_id=<?= $turma_id ?>&trimestre=<?= $trimestre ?>" class="btn-back">
    &leftarrow; Voltar à Estatística por Disciplina
</a>

<div class="container">
    <?php if ($turma_id && $trimestre): ?>
        <div class="cabecalho">
            <h2>ESTATÍSTICA GLOBAL DA TURMA</h2>
            <h3>RESUMO GERAL DE TRANSIÇÃO E REPROVAÇÃO</h3>
        </div>

        <div class="info-trimestre-turma">
            <?= htmlspecialchars($trimestre_extenso) ?> - Turma: **<?= htmlspecialchars($classe_turma) ?>** - **<?= htmlspecialchars($nome_turma) ?>** - Curso: **<?= htmlspecialchars($curso_turma) ?>** - Ano Letivo: **<?= htmlspecialchars($ano_lectivo) ?>**
        </div>

        <table>
            <thead>
                <tr>
                    <th rowspan="2">INDICADOR</th>
                    <th colspan="3">GÊNERO</th>
                </tr>
                <tr>
                    <th>Masculino</th>
                    <th>Feminino</th>
                    <th>TOTAL</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td style="text-align: left;">Alunos Matriculados</td>
                    <td><?= $estatisticas_gerais['matriculados']['M'] ?></td>
                    <td><?= $estatisticas_gerais['matriculados']['F'] ?></td>
                    <td class="col-total resultado"><?= $estatisticas_gerais['matriculados']['Total'] ?></td>
                </tr>
                <tr>
                    <td style="text-align: left;">Abandono</td>
                    <td><?= $estatisticas_gerais['abandono']['M'] ?></td>
                    <td><?= $estatisticas_gerais['abandono']['F'] ?></td>
                    <td class="col-total" style="background-color: #ffe6e6;"><?= $estatisticas_gerais['abandono']['Total'] ?></td>
                </tr>
                <tr>
                    <td style="text-align: left;">Transferidos</td>
                    <td><?= $estatisticas_gerais['transferido']['M'] ?></td>
                    <td><?= $estatisticas_gerais['transferido']['F'] ?></td>
                    <td class="col-total" style="background-color: #fff0e6;"><?= $estatisticas_gerais['transferido']['Total'] ?></td>
                </tr>
                <tr style="background-color: #dbeaff;">
                    <?php 
                        $efetivo_m = $estatisticas_gerais['matriculados']['M'] - $estatisticas_gerais['abandono']['M'] - $estatisticas_gerais['transferido']['M'];
                        $efetivo_f = $estatisticas_gerais['matriculados']['F'] - $estatisticas_gerais['abandono']['F'] - $estatisticas_gerais['transferido']['F'];
                        $efetivo_total = $matriculados_total_efetivo;
                    ?>
                    <td style="text-align: left; font-weight: bold;">TOTAL EFETIVO (Para Aprovação)</td>
                    <td class="resultado"><?= $efetivo_m ?></td>
                    <td class="resultado"><?= $efetivo_f ?></td>
                    <td class="col-total resultado"><?= $efetivo_total ?></td>
                </tr>
                <tr style="background-color: #e6ffe6;">
                    <td style="text-align: left; color: #28a745; font-weight: bold;">Alunos Aprovados (Média Geral ≥ 10)</td>
                    <td class="resultado" style="color: #28a745;"><?= $estatisticas_gerais['aprovados']['M'] ?></td>
                    <td class="resultado" style="color: #28a745;"><?= $estatisticas_gerais['aprovados']['F'] ?></td>
                    <td class="col-total resultado" style="color: #28a745;"><?= $estatisticas_gerais['aprovados']['Total'] ?></td>
                </tr>
                <tr style="background-color: #ffe6e6;">
                    <td style="text-align: left; color: #dc3545; font-weight: bold;">Alunos Reprovados (Média Geral < 10)</td>
                    <td class="resultado" style="color: #dc3545;"><?= $estatisticas_gerais['reprovados']['M'] ?></td>
                    <td class="resultado" style="color: #dc3545;"><?= $estatisticas_gerais['reprovados']['F'] ?></td>
                    <td class="col-total resultado" style="color: #dc3545;"><?= $estatisticas_gerais['reprovados']['Total'] ?></td>
                </tr>
                
                <tr style="border-top: 3px double #333;">
                    <td style="text-align: left; font-weight: bold;">Aproveitamento Efetivo (%)</td>
                    <?php 
                        $aprov_m = $efetivo_m > 0 ? round(($estatisticas_gerais['aprovados']['M'] / $efetivo_m) * 100, 1) : 0;
                        $aprov_f = $efetivo_f > 0 ? round(($estatisticas_gerais['aprovados']['F'] / $efetivo_f) * 100, 1) : 0;
                    ?>
                    <td class="percentagem"><?= $aprov_m ?>%</td>
                    <td class="percentagem"><?= $aprov_f ?>%</td>
                    <td class="col-total percentagem"><?= $aproveitamento_geral ?>%</td>
                </tr>
                <tr>
                    <td style="text-align: left; font-weight: bold;">Rendimento Global (%)</td>
                    <?php 
                        $rend_m = $estatisticas_gerais['matriculados']['M'] > 0 ? round(($estatisticas_gerais['aprovados']['M'] / $estatisticas_gerais['matriculados']['M']) * 100, 1) : 0;
                        $rend_f = $estatisticas_gerais['matriculados']['F'] > 0 ? round(($estatisticas_gerais['aprovados']['F'] / $estatisticas_gerais['matriculados']['F']) * 100, 1) : 0;
                    ?>
                    <td class="percentagem"><?= $rend_m ?>%</td>
                    <td class="percentagem"><?= $rend_f ?>%</td>
                    <td class="col-total percentagem"><?= $rendimento_geral ?>%</td>
                </tr>
            </tbody>
        </table>

        <div class="data-mapa">
            Catumbela, aos <?= $data_atual ?>
        </div>

        <div class="assinaturas">
            <div class="assinatura-esquerda">
                <p>Assinatura do Director de Turma</p>
                <div class="line"></div>
            </div>
            <div class="assinatura-direita">
                <p>Assinatura do Subdirecto Pedagógico</p>
                <div class="line"></div>
            </div>
        </div>

    <?php else: ?>
        <div style="text-align: center; margin-top: 20px; font-size: 1.1em; color: #cc3300;">
            Selecione uma Turma e um Trimestre para ver a Estatística Global.
        </div>
    <?php endif; ?>
</div>

</body>
</html>