<?php
// Inicia a sessão e inclui arquivos de configuração.
session_start();
require_once 'conexao.php';
// Adicione 'verifica_login.php' se ele contiver a lógica de autenticação.
// require_once 'verifica_login.php'; 

// --- Autenticação e Dados do Usuário ---
if (!isset($_SESSION['usuario']) || $_SESSION['usuario']['perfil'] !== 'PROFESSOR') {
    // Redireciona para o login se não for professor (ajuste conforme seu sistema)
    header("Location: login.php"); 
    exit;
}

$usuario = $_SESSION['usuario'];
$usuario_id = intval($usuario['id']);
$nome = htmlspecialchars($usuario['nome'] ?? 'Professor');

// --- Recebe Parâmetros de Filtro (GET) ---
$disciplina_id_selecionada = filter_input(INPUT_GET, 'disciplina_id', FILTER_VALIDATE_INT);
$trimestre_selecionado = filter_input(INPUT_GET, 'trimestre', FILTER_VALIDATE_INT);

$estatisticas_disciplina = [];
$nome_disciplina = '';
$nome_turma_disciplina = '';
$classe_turma_disciplina = '';
$trimestre_extenso = '';

// Mapeamento do trimestre para exibição
if ($trimestre_selecionado == 1) {
    $trimestre_extenso = 'I Trimestre';
} elseif ($trimestre_selecionado == 2) {
    $trimestre_extenso = 'II Trimestre';
} elseif ($trimestre_selecionado == 3) {
    $trimestre_extenso = 'III Trimestre';
}

// --- Busca Inicial: Disciplinas do Professor ---
$disciplinas_professor = [];
$query_disciplinas = "
    SELECT 
        d.id AS disciplina_id, 
        d.nome AS nome_disciplina, 
        t.id AS turma_id, 
        t.nome AS nome_turma, 
        t.classe, 
        t.curso
    FROM disciplinas d
    JOIN turmas t ON d.turma_id = t.id
    JOIN atribuicoes a ON d.turma_id = a.turma_id AND a.disciplina_id = d.id
    WHERE a.professor_id = ?
    ORDER BY t.classe, t.nome, d.nome
";
$stmt_disciplinas = $conn->prepare($query_disciplinas);
$stmt_disciplinas->bind_param("i", $usuario_id);
$stmt_disciplinas->execute();
$result_disciplinas = $stmt_disciplinas->get_result();

while ($row = $result_disciplinas->fetch_assoc()) {
    $row['nome_completo'] = htmlspecialchars("{$row['nome_disciplina']} ({$row['classe']} - {$row['nome_turma']})");
    $disciplinas_professor[] = $row;
}
$stmt_disciplinas->close();

// --- Lógica de Cálculo de Estatísticas por Disciplina e Turma ---
$data_aproveitamento_grafico = []; // Para o gráfico de Aproveitamento

if ($disciplina_id_selecionada && $trimestre_selecionado) {
    
    // 1. Busca a disciplina e turma selecionada
    $stmt_info = $conn->prepare("
        SELECT d.nome AS nome_disciplina, t.nome AS nome_turma, t.classe, t.id AS turma_id
        FROM disciplinas d
        JOIN turmas t ON d.turma_id = t.id
        WHERE d.id = ?
    ");
    $stmt_info->bind_param("i", $disciplina_id_selecionada);
    $stmt_info->execute();
    $info = $stmt_info->get_result()->fetch_assoc();
    $stmt_info->close();

    if ($info) {
        $nome_disciplina = htmlspecialchars($info['nome_disciplina']);
        $nome_turma_disciplina = htmlspecialchars($info['nome_turma']);
        $classe_turma_disciplina = htmlspecialchars($info['classe']);
        $turma_id = $info['turma_id'];
        
        // 2. Busca todos os alunos matriculados nesta turma
        $stmt_alunos = $conn->prepare("SELECT id, nome, genero FROM alunos WHERE turma_id = ? ORDER BY nome");
        $stmt_alunos->bind_param("i", $turma_id);
        $stmt_alunos->execute();
        $alunos = $stmt_alunos->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt_alunos->close();

        // Inicializa contadores
        $estatisticas_disciplina = [
            'matriculados' => ['M' => 0, 'F' => 0, 'Total' => 0],
            'avaliados' => ['M' => 0, 'F' => 0, 'Total' => 0],
            'aprovados' => ['M' => 0, 'F' => 0, 'Total' => 0],
            'reprovados' => ['M' => 0, 'F' => 0, 'Total' => 0],
            'aproveitamento_perc' => ['M' => 0, 'F' => 0, 'Total' => 0],
            'rendimento_perc' => ['M' => 0, 'F' => 0, 'Total' => 0],
        ];

        // Prepara statements para notas
        $stmt_notas = $conn->prepare("
            SELECT 
                n.npp, n.npt, 
                ac.mac 
            FROM notas n
            LEFT JOIN avaliacoes_continuas ac ON n.aluno_id = ac.aluno_id AND n.disciplina_id = ac.disciplina_id AND n.trimestre = ac.trimestre
            WHERE n.aluno_id = ? AND n.disciplina_id = ? AND n.trimestre = ?
        ");
        
        // Loop pelos alunos
        foreach ($alunos as $aluno) {
            $genero_col = $aluno['genero'] ?? 'M'; 
            $genero = strtoupper(substr($genero_col, 0, 1));
            $genero = ($genero === 'F') ? 'F' : 'M';

            // Contabiliza matriculados
            $estatisticas_disciplina['matriculados'][$genero]++;
            $estatisticas_disciplina['matriculados']['Total']++;

            $mac = $npp = $npt = $media_trimestral = 0;

            // Busca as notas
            $stmt_notas->bind_param("iii", $aluno['id'], $disciplina_id_selecionada, $trimestre_selecionado);
            $stmt_notas->execute();
            $res_notas = $stmt_notas->get_result()->fetch_assoc();
            
            if ($res_notas) {
                $mac = floatval($res_notas['mac'] ?? 0);
                $npp = floatval($res_notas['npp'] ?? 0);
                $npt = floatval($res_notas['npt'] ?? 0);
            }

            // Cálculo da Média Trimestral
            if ($mac > 0 || $npp > 0 || $npt > 0) {
                $media_trimestral = ($mac + $npp + $npt) / 3;
                
                // Contabiliza Avaliados
                $estatisticas_disciplina['avaliados'][$genero]++;
                $estatisticas_disciplina['avaliados']['Total']++;

                // Aprovação: Média arredondada >= 10
                if (round($media_trimestral) >= 10) {
                    $estatisticas_disciplina['aprovados'][$genero]++;
                    $estatisticas_disciplina['aprovados']['Total']++;
                } else {
                    $estatisticas_disciplina['reprovados'][$genero]++;
                    $estatisticas_disciplina['reprovados']['Total']++;
                }
            }
        } // Fim Loop Alunos
        
        $stmt_notas->close();
        
        // 3. Cálculo de Percentagens
        foreach (['M', 'F', 'Total'] as $g) {
            $aprovados = $estatisticas_disciplina['aprovados'][$g];
            $avaliados = $estatisticas_disciplina['avaliados'][$g];
            $matriculados = $estatisticas_disciplina['matriculados'][$g];

            // Aproveitamento (%): Aprovados / Avaliados
            if ($avaliados > 0) {
                $aproveitamento_perc = round(($aprovados / $avaliados) * 100, 1);
                $estatisticas_disciplina['aproveitamento_perc'][$g] = $aproveitamento_perc;
            } else {
                $aproveitamento_perc = 0;
            }

            // Rendimento (%): Aprovados / Matriculados
            if ($matriculados > 0) {
                $estatisticas_disciplina['rendimento_perc'][$g] = round(($aprovados / $matriculados) * 100, 1);
            }
            
            // Dados para o Gráfico de Aproveitamento (usando Avaliados)
            if ($g === 'Total') {
                 $data_aproveitamento_grafico[] = $avaliados > 0 ? $aproveitamento_perc : 0;
            }
        }
    }
}

// Fecha a conexão com o banco de dados.
$conn->close();
?>
<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Estatística por Disciplina - Professor</title>
    <link rel="stylesheet" href="css/all.min.css">
    <link rel="stylesheet" href="css/inter.css">
    <script src="js/tailwindcss.js"></script>
    <script src="js/chart.umd.min.js"></script>
    <style>
        /* Estilos base (mantidos do painel) */
        body { font-family: 'Inter', sans-serif; }
        .sidebar {
            transition: transform 0.3s ease-in-out;
            transform: translateX(-100%);
            z-index: 999;
        }
        .sidebar.active { transform: translateX(0); }
        .overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 998;
            display: none;
        }
        .overlay.active { display: block; }
        @media (min-width: 768px) {
            .sidebar { transform: translateX(0); }
            .content { margin-left: 16rem; }
            .overlay { display: none !important; }
        }
        .mensagem { /* ... (mantido) ... */ }
        
        /* Estilos específicos da tabela de estatística */
        .estatistica-table th, .estatistica-table td {
            border: 1px solid #e5e7eb;
            padding: 10px;
            text-align: center;
        }
        .estatistica-table th {
            background-color: #1f2937;
            color: white;
            font-weight: 600;
        }
        .estatistica-table .metric-header {
            background-color: #374151;
            color: #d1d5db;
        }
        .estatistica-table .total-col {
            background-color: #e5e7eb;
            font-weight: bold;
        }
    </style>
</head>
<body class="bg-gray-100 min-h-screen flex flex-col">

    <header class="fixed top-0 left-0 w-full md:left-64 md:w-[calc(100%-16rem)] h-16 bg-blue-900 text-white flex items-center justify-between p-4 shadow-lg z-50">
        <button id="menu-toggle" class="md:hidden text-2xl focus:outline-none">
            <i class="fas fa-bars"></i>
        </button>
        <div class="flex items-center gap-4 ml-auto">
            <span class="hidden sm:block font-bold"><?= $nome ?></span>
            <img src="fotos/<?= $usuario['foto'] ?? 'sem-foto.png' ?>" alt="Foto do Professor" class="w-10 h-10 rounded-full border-2 border-white object-cover">
        </div>
    </header>

    <div id="overlay" class="overlay"></div>

    <nav id="sidebar" class="sidebar fixed top-0 left-0 w-64 h-full bg-blue-900 text-white p-4 pt-16 shadow-lg overflow-y-auto">
        <div class="flex flex-col items-center justify-center p-4">
            <img src="fotos/<?= $usuario['foto'] ?? 'sem-foto.png' ?>" alt="Foto do Professor" class="w-20 h-20 rounded-full border-4 border-white mb-2 object-cover">
            <span class="text-xl font-bold text-center"><?= $nome ?></span>
        </div>
        <ul class="mt-8 space-y-2">
            <li><a href="painel_professor1.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-tachometer-alt"></i><span>Início</span></a></li>
            <li><a href="mini_pauta_professor1.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-pen"></i><span>Mini Pauta</span></a></li>
            <li><a href="caderneta_professor.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-book"></i><span>Caderneta</span></a></li>
            <li><a href="estatistica_professor.php" class="flex items-center gap-3 p-3 rounded-lg bg-blue-700 transition duration-200"><i class="fas fa-chart-bar"></i><span>Estatística</span></a></li>
            <li><a href="historico_lancamentos.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-history"></i><span>Histórico Lançamentos</span></a></li>
            <li><a href="login.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-sign-out-alt"></i><span>Sair</span></a></li>
        </ul>
    </nav>

    <main class="content flex-grow p-5 md:ml-64 mt-16">
        <h1 class="text-3xl font-bold text-gray-800 mb-6">📊 Estatística por Disciplina</h1>

        <section class="bg-white p-6 rounded-xl shadow-md mb-6">
            <h2 class="text-xl font-semibold text-gray-700 mb-4">Selecione a Disciplina e o Trimestre</h2>
            <form method="get" class="flex flex-col sm:flex-row gap-4 items-end">
                <div class="flex-1 w-full sm:w-auto">
                    <label for="disciplina_id" class="block text-sm font-medium text-gray-700">Disciplina/Turma:</label>
                    <select name="disciplina_id" id="disciplina_id" class="mt-1 block w-full pl-3 pr-10 py-2 text-base border-gray-300 focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm rounded-md" required>
                        <option value="">-- Selecionar Disciplina/Turma --</option>
                        <?php foreach ($disciplinas_professor as $disc): ?>
                            <option value="<?= $disc['disciplina_id'] ?>" <?= ($disc['disciplina_id'] == $disciplina_id_selecionada) ? 'selected' : '' ?>>
                                <?= $disc['nome_completo'] ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="w-full sm:w-auto">
                    <label for="trimestre" class="block text-sm font-medium text-gray-700">Trimestre:</label>
                    <select name="trimestre" id="trimestre" class="mt-1 block w-full pl-3 pr-10 py-2 text-base border-gray-300 focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm rounded-md" required>
                        <option value="1" <?= ($trimestre_selecionado == 1) ? 'selected' : '' ?>>I Trimestre</option>
                        <option value="2" <?= ($trimestre_selecionado == 2) ? 'selected' : '' ?>>II Trimestre</option>
                        <option value="3" <?= ($trimestre_selecionado == 3) ? 'selected' : '' ?>>III Trimestre</option>
                    </select>
                </div>
                
                <button type="submit" class="w-full sm:w-auto bg-blue-600 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded transition duration-200">
                    Ver Estatística
                </button>
            </form>
            
            <?php if (empty($disciplinas_professor)): ?>
                <p class="mt-4 text-red-500">Você não tem disciplinas atribuídas para gerar estatísticas.</p>
            <?php endif; ?>
        </section>

        <?php if ($disciplina_id_selecionada && $trimestre_selecionado && !empty($estatisticas_disciplina)): ?>
        
        <section class="bg-white p-6 rounded-xl shadow-md mb-6">
            <h2 class="text-2xl font-bold text-gray-800 mb-2">Estatística da Disciplina: <?= $nome_disciplina ?></h2>
            <p class="text-gray-600 mb-4">Turma: <?= $classe_turma_disciplina ?> - <?= $nome_turma_disciplina ?> | Período: <?= $trimestre_extenso ?></p>

            <div class="overflow-x-auto">
                <table class="w-full estatistica-table">
                    <thead>
                        <tr>
                            <th class="metric-header" colspan="2">Métrica</th>
                            <th>Masculino (M)</th>
                            <th>Feminino (F)</th>
                            <th class="total-col">Total</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td rowspan="2" class="metric-header font-bold text-left">ALUNOS</td>
                            <td class="text-left">Matriculados</td>
                            <td><?= $estatisticas_disciplina['matriculados']['M'] ?></td>
                            <td><?= $estatisticas_disciplina['matriculados']['F'] ?></td>
                            <td class="total-col"><?= $estatisticas_disciplina['matriculados']['Total'] ?></td>
                        </tr>
                        <tr>
                            <td class="text-left">Avaliados</td>
                            <td><?= $estatisticas_disciplina['avaliados']['M'] ?></td>
                            <td><?= $estatisticas_disciplina['avaliados']['F'] ?></td>
                            <td class="total-col"><?= $estatisticas_disciplina['avaliados']['Total'] ?></td>
                        </tr>
                        <tr>
                            <td rowspan="2" class="metric-header font-bold text-left">RESULTADO</td>
                            <td class="text-left">Aprovados</td>
                            <td><?= $estatisticas_disciplina['aprovados']['M'] ?></td>
                            <td><?= $estatisticas_disciplina['aprovados']['F'] ?></td>
                            <td class="total-col"><?= $estatisticas_disciplina['aprovados']['Total'] ?></td>
                        </tr>
                        <tr>
                            <td class="text-left">Reprovados</td>
                            <td><?= $estatisticas_disciplina['reprovados']['M'] ?></td>
                            <td><?= $estatisticas_disciplina['reprovados']['F'] ?></td>
                            <td class="total-col"><?= $estatisticas_disciplina['reprovados']['Total'] ?></td>
                        </tr>
                        <tr>
                            <td rowspan="2" class="metric-header font-bold text-left">PERCENTUAL</td>
                            <td class="text-left">Aproveitamento (%) (Aprovados / Avaliados)</td>
                            <td><?= $estatisticas_disciplina['aproveitamento_perc']['M'] ?>%</td>
                            <td><?= $estatisticas_disciplina['aproveitamento_perc']['F'] ?>%</td>
                            <td class="total-col"><?= $estatisticas_disciplina['aproveitamento_perc']['Total'] ?>%</td>
                        </tr>
                        <tr>
                            <td class="text-left">Rendimento (%) (Aprovados / Matriculados)</td>
                            <td><?= $estatisticas_disciplina['rendimento_perc']['M'] ?>%</td>
                            <td><?= $estatisticas_disciplina['rendimento_perc']['F'] ?>%</td>
                            <td class="total-col"><?= $estatisticas_disciplina['rendimento_perc']['Total'] ?>%</td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </section>
        
        <section class="bg-white p-6 rounded-xl shadow-md mb-6">
            <h2 class="text-xl font-semibold text-gray-700 mb-4">Gráfico de Aproveitamento por Gênero</h2>
            <div class="relative h-80 max-w-lg mx-auto">
                <canvas id="graficoAproveitamentoDisciplina"></canvas>
            </div>
        </section>

        <?php elseif ($disciplina_id_selecionada && $trimestre_selecionado): ?>
             <div class="bg-yellow-100 border-l-4 border-yellow-500 text-yellow-700 p-4" role="alert">
                <p class="font-bold">Dados não encontrados</p>
                <p>Nenhum aluno desta disciplina teve notas lançadas para o **<?= $trimestre_extenso ?>**.</p>
            </div>
        <?php endif; ?>

    </main>

    <script>
        // Lógica para alternar a barra lateral
        const menuToggle = document.getElementById('menu-toggle');
        const sidebar = document.getElementById('sidebar');
        const overlay = document.getElementById('overlay');

        function toggleSidebar() {
            sidebar.classList.toggle('active');
            overlay.classList.toggle('active');
        }

        menuToggle.addEventListener('click', toggleSidebar);
        overlay.addEventListener('click', toggleSidebar);

        <?php if ($disciplina_id_selecionada && $trimestre_selecionado && !empty($estatisticas_disciplina)): ?>
        
        // Dados para o Gráfico de Aproveitamento
        const aproveitamentoData = {
            labels: ['Masculino', 'Feminino', 'Total'],
            datasets: [{
                label: 'Aproveitamento (%) (Aprovados/Avaliados)',
                data: [
                    <?= $estatisticas_disciplina['aproveitamento_perc']['M'] ?>, 
                    <?= $estatisticas_disciplina['aproveitamento_perc']['F'] ?>, 
                    <?= $estatisticas_disciplina['aproveitamento_perc']['Total'] ?>
                ],
                backgroundColor: [
                    'rgba(0, 123, 255, 0.6)', // Azul (M)
                    'rgba(255, 99, 132, 0.6)', // Vermelho/Rosa (F)
                    'rgba(40, 167, 69, 0.6)' // Verde (Total)
                ],
                borderColor: [
                    'rgba(0, 123, 255, 1)',
                    'rgba(255, 99, 132, 1)',
                    'rgba(40, 167, 69, 1)'
                ],
                borderWidth: 1
            }]
        };

        const ctxAproveitamento = document.getElementById('graficoAproveitamentoDisciplina');
        new Chart(ctxAproveitamento, {
            type: 'bar',
            data: aproveitamentoData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false },
                    title: {
                        display: true,
                        text: 'Aproveitamento em <?= $nome_disciplina ?> - <?= $trimestre_extenso ?>'
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        max: 100,
                        title: { display: true, text: 'Percentual (%)' }
                    }
                }
            }
        });
        <?php endif; ?>
    </script>
</body>
</html>