<?php
include 'conexao.php';
session_start();

// Adiciona esta linha para garantir a codificação UTF-8
if (isset($conn)) {
    $conn->set_charset("utf8");
}

// Verificação de sessão e permissões
if (!isset($_SESSION['usuario']) || !in_array(strtoupper($_SESSION['usuario']['perfil']), ['SUBDIRECÇÃO PEDAGÓGICA', 'ADMINISTRADOR'])) {
    header("Location: login.php");
    exit;
}

$perfil_gestor = strtoupper($_SESSION['usuario']['perfil']);
$ano_atual = date('Y'); // Ex: '2025'

// --- Lógica de Seleção de Trimestre ---
$trimestre_ativo = filter_input(INPUT_GET, 'trimestre', FILTER_VALIDATE_INT);
if (!$trimestre_ativo || $trimestre_ativo < 1 || $trimestre_ativo > 3) {
    $trimestre_ativo = 1; // Padrão é o 1º Trimestre
}

// --- 1. Buscar todas as atribuições (Professor x Turma x Disciplina) ---
$atribuicoes_detalhadas = [];

$query_atribuicoes = "
    SELECT 
        u.id AS professor_id,
        u.nome AS nome_professor,
        atr.turma_id,
        atr.disciplina_id,
        t.classe,
        t.nome AS nome_turma,
        t.curso, 
        d.nome AS nome_disciplina
    FROM atribuicoes atr
    JOIN usuarios u ON atr.professor_id = u.id
    JOIN turmas t ON atr.turma_id = t.id
    JOIN disciplinas d ON atr.disciplina_id = d.id
    WHERE u.perfil = 'PROFESSOR'
    ORDER BY u.nome ASC, t.classe ASC, t.nome ASC
";

$result_atribuicoes = $conn->query($query_atribuicoes);

if ($result_atribuicoes && $result_atribuicoes->num_rows > 0) {
    while ($row = $result_atribuicoes->fetch_assoc()) {
        $atribuicoes_detalhadas[] = $row;
    }
}

// --- 2. Buscar o status de lançamento de notas e DATA MAIS RECENTE (Lógica de Cobertura) ---
foreach ($atribuicoes_detalhadas as &$atribuicao) {
    $turma_id = $atribuicao['turma_id'];
    $disciplina_id = $atribuicao['disciplina_id'];

    // 2.1. CONTAGEM DE ALUNOS ATIVOS NA TURMA 
    // CRITÉRIO CORRIGIDO: Inclui todos, exceto se observacoes na tabela ALUNOS contiver 'Desistente'
    $query_total_alunos = "
        SELECT COUNT(m.id) AS total_alunos 
        FROM matriculas m
        JOIN alunos a ON m.aluno_id = a.id -- JOIN com alunos para acessar a.observacoes
        WHERE m.turma_id = ? 
          AND m.ano_letivo = ?
          -- NOVO FILTRO (Definitivo): Aluno ativo se a.observacoes for NULL/Vazio OU não contiver 'Desistente'
          AND (TRIM(a.observacoes) IS NULL OR TRIM(a.observacoes) = '' OR a.observacoes NOT LIKE '%Desistente%')
    ";
    $stmt_total = $conn->prepare($query_total_alunos);
    $stmt_total->bind_param("is", $turma_id, $ano_atual);
    $stmt_total->execute();
    $result_total = $stmt_total->get_result();
    $total_alunos = $result_total->fetch_assoc()['total_alunos'];
    $stmt_total->close();

    // 2.2. Se não houver alunos ativos, define como N/A e pula a verificação
    if ($total_alunos == 0) {
        $atribuicao['total_alunos'] = 0; 
        $atribuicao['data_lancamento'] = 'N/A';
        $atribuicao['status_mac'] = 'N/A';
        $atribuicao['status_npp'] = 'N/A';
        $atribuicao['status_npt'] = 'N/A';
        $atribuicao['status_mt'] = 'N/A';
        continue;
    }
    $atribuicao['total_alunos'] = $total_alunos; 

    // 2.3. CONTAGEM DE LANÇAMENTOS VÁLIDOS E DATA MAIS RECENTE
    $query_status = "
        SELECT 
            MAX(n.data_lancamento) AS data_lancamento_mais_recente,
            SUM(CASE WHEN n.mac IS NOT NULL AND n.mac >= 0 THEN 1 ELSE 0 END) AS lancados_mac,
            SUM(CASE WHEN n.npp IS NOT NULL AND n.npp >= 0 THEN 1 ELSE 0 END) AS lancados_npp,
            SUM(CASE WHEN n.npt IS NOT NULL AND n.npt >= 0 THEN 1 ELSE 0 END) AS lancados_npt,
            SUM(CASE WHEN n.mt IS NOT NULL AND n.mt >= 0 THEN 1 ELSE 0 END) AS lancados_mt
        FROM notas n
        -- JOIN com a MATRICULAS para filtrar por TURMA e ANO
        JOIN matriculas m ON n.aluno_id = m.aluno_id 
                         AND m.turma_id = ? 
                         AND m.ano_letivo = ? 
        -- JOIN com ALUNOS para aplicar o filtro de Desistente
        JOIN alunos a ON m.aluno_id = a.id 
        
        WHERE n.turma_id = ? 
          AND n.disciplina_id = ? 
          AND n.trimestre = ?
          -- NOVO FILTRO (Definitivo): Aluno ativo se a.observacoes for NULL/Vazio OU não contiver 'Desistente'
          AND (TRIM(a.observacoes) IS NULL OR TRIM(a.observacoes) = '' OR a.observacoes NOT LIKE '%Desistente%')
    ";
    
    $stmt_status = $conn->prepare($query_status);
    // bind_param para: turma_id(i), ano_atual(s), turma_id(i), disciplina_id(i), trimestre_ativo(i)
    $stmt_status->bind_param("isiii", $turma_id, $ano_atual, $turma_id, $disciplina_id, $trimestre_ativo);
    $stmt_status->execute();
    $result_status = $stmt_status->get_result();

    if ($result_status->num_rows > 0) {
        $status = $result_status->fetch_assoc();
        
        // Determina a data
        $atribuicao['data_lancamento'] = $status['data_lancamento_mais_recente'] 
            ? date('d/m/Y H:i', strtotime($status['data_lancamento_mais_recente'])) 
            : 'N/A';

        // Função para determinar o status (✅, ⚠️, ❌)
        $determinar_status = function($lancados) use ($total_alunos) {
            if ($total_alunos == 0) {
                 return 'N/A'; // Nenhuma nota é esperada
            } elseif ($lancados == $total_alunos) {
                return '✅'; // Completo
            } elseif ($lancados > 0) {
                return '⚠️'; // Incompleto, mas algo foi lançado
            } else {
                return '❌'; // Nada lançado
            }
        };

        $atribuicao['status_mac'] = $determinar_status($status['lancados_mac']);
        $atribuicao['status_npp'] = $determinar_status($status['lancados_npp']);
        $atribuicao['status_npt'] = $determinar_status($status['lancados_npt']);
        $atribuicao['status_mt'] = $determinar_status($status['lancados_mt']);
        
    } else {
        $atribuicao['data_lancamento'] = 'N/A';
        $atribuicao['status_mac'] = '❌';
        $atribuicao['status_npp'] = '❌';
        $atribuicao['status_npt'] = '❌';
        $atribuicao['status_mt'] = '❌';
    }
    $stmt_status->close();
}
unset($atribuicao); 

$conn->close();
?>
<!DOCTYPE html>
<html lang="pt">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Controlo de Históricos - SISMAG</title>
<link rel="stylesheet" href="https://rsms.me/inter/inter.css">
<script src="https://cdn.tailwindcss.com"></script>
<link rel="stylesheet" href="css/all.min.css">
<style>
    body { font-family: 'Inter', sans-serif; }
    .status-cell {
        text-align: center;
        font-size: 1.1em;
        font-weight: bold;
    }
    /* Cores para os status (usadas via título no span) */
    .status-cell span[title*="Lançado Completo"] { color: #10B981; } /* Verde */
    .status-cell span[title*="Lançamento Incompleto"] { color: #F59E0B; } /* Laranja */
    .status-cell span[title*="Pendente"] { color: #EF4444; } /* Vermelho */
    .status-cell span[title*="N/A"] { color: #9CA3AF; } /* Cinza */
</style>
</head>
<body class="bg-gray-100 flex min-h-screen">

<aside class="w-60 bg-[#0d1b2a] text-white flex flex-col">
    <div class="p-4 text-center border-b border-gray-700">
        <h1 class="text-xl font-bold text-blue-400">SISMAG</h1>
        <p class="text-xs text-gray-400 mt-1"><?= $perfil_gestor ?></p>
    </div>

    <nav class="flex-1 mt-3">
        <a href="painel_gestor.php" class="block px-5 py-3 hover:bg-[#1b263b] transition">
            <i class="fas fa-home text-blue-400 mr-2"></i> Início
        </a>
        <a href="estatistica_turma.php" class="block px-5 py-3 hover:bg-[#1b263b] transition">
            <i class="fas fa-chart-bar text-blue-400 mr-2"></i> Estatísticas
        </a>
        <a href="historico_controlo.php" class="block px-5 py-3 bg-[#1b263b] transition">
            <i class="fas fa-list-ul text-blue-400 mr-2"></i> Histórico
        </a>
        <a href="logout.php" class="block px-5 py-3 hover:bg-red-700 transition">
            <i class="fas fa-sign-out-alt text-red-400 mr-2"></i> Sair
        </a>
    </nav>
</aside>

<main class="flex-1 flex flex-col">
    <header class="bg-blue-700 text-white p-4 shadow flex justify-between items-center">
        <h2 class="text-lg font-semibold flex items-center gap-2">
            <i class="fas fa-list-ul text-white"></i> Controlo de Lançamentos (<?= $perfil_gestor ?>)
        </h2>
        <a href="painel_gestor.php" class="text-sm hover:text-gray-200 transition">
            <i class="fas fa-arrow-left"></i> Voltar
        </a>
    </header>

    <section class="flex-1 p-6">
        <div class="bg-white rounded-xl shadow-lg p-6 mx-auto">
            
            <form method="GET" class="mb-6 flex items-end gap-4 border-b pb-4">
                <div class="flex-grow max-w-xs">
                    <label for="trimestre" class="block text-sm font-medium text-gray-700 mb-1">Selecione o Trimestre de Monitoramento</label>
                    <select id="trimestre" name="trimestre" class="mt-1 block w-full pl-3 pr-10 py-2 text-base border-gray-300 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm rounded-md">
                        <option value="1" <?= $trimestre_ativo == 1 ? 'selected' : '' ?>>1º Trimestre</option>
                        <option value="2" <?= $trimestre_ativo == 2 ? 'selected' : '' ?>>2º Trimestre</option>
                        <option value="3" <?= $trimestre_ativo == 3 ? 'selected' : '' ?>>3º Trimestre</option>
                    </select>
                </div>
                <button type="submit" class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 transition duration-150">
                    <i class="fas fa-filter mr-2"></i> Filtrar
                </button>
            </form>

            <h3 class="text-xl font-semibold text-gray-800 mb-4">Monitoramento de Lançamentos - <span class="text-blue-600"><?= $trimestre_ativo ?>º Trimestre</span></h3>
            
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Professor</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Atribuição (Turma/Curso)</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Disciplina</th>
                            <th class="px-3 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider" title="Data do Lançamento mais Recente">Data Recente</th>
                            <th class="px-3 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">MAC</th>
                            <th class="px-3 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">NPP</th>
                            <th class="px-3 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">NPT</th>
                            <th class="px-3 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">MT</th>
                            <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">Histórico</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        <?php if (empty($atribuicoes_detalhadas)): ?>
                            <tr>
                                <td colspan="9" class="px-6 py-4 text-center text-sm text-gray-500">
                                    Nenhuma atribuição de professor/turma/disciplina encontrada.
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($atribuicoes_detalhadas as $atribuicao): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                                        <?= htmlspecialchars($atribuicao['nome_professor']) ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-700" title="<?= htmlspecialchars($atribuicao['curso']) ?>">
                                        <?= htmlspecialchars($atribuicao['classe']) . 'ª-' . htmlspecialchars($atribuicao['nome_turma']) . ' - ' . htmlspecialchars($atribuicao['curso']) . " ({$atribuicao['total_alunos']} alunos)" ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-700">
                                        <?= htmlspecialchars($atribuicao['nome_disciplina']) ?>
                                    </td>
                                    
                                    <td class="px-3 py-4 text-sm text-center text-gray-500">
                                        <?= $atribuicao['data_lancamento'] ?>
                                    </td>
                                    <td class="px-3 py-4 status-cell">
                                        <span title="MAC: <?= $atribuicao['status_mac'] == '✅' ? 'Lançado Completo' : ($atribuicao['status_mac'] == '⚠️' ? 'Lançamento Incompleto' : ($atribuicao['status_mac'] == 'N/A' ? 'N/A' : 'Pendente')) ?>"><?= $atribuicao['status_mac'] ?></span>
                                    </td>
                                    <td class="px-3 py-4 status-cell">
                                        <span title="NPP: <?= $atribuicao['status_npp'] == '✅' ? 'Lançado Completo' : ($atribuicao['status_npp'] == '⚠️' ? 'Lançamento Incompleto' : ($atribuicao['status_npp'] == 'N/A' ? 'N/A' : 'Pendente')) ?>"><?= $atribuicao['status_npp'] ?></span>
                                    </td>
                                    <td class="px-3 py-4 status-cell">
                                        <span title="NPT: <?= $atribuicao['status_npt'] == '✅' ? 'Lançado Completo' : ($atribuicao['status_npt'] == '⚠️' ? 'Lançamento Incompleto' : ($atribuicao['status_npt'] == 'N/A' ? 'N/A' : 'Pendente')) ?>"><?= $atribuicao['status_npt'] ?></span>
                                    </td>
                                    <td class="px-3 py-4 status-cell">
                                        <span title="MT: <?= $atribuicao['status_mt'] == '✅' ? 'Lançado Completo' : ($atribuicao['status_mt'] == '⚠️' ? 'Lançamento Incompleto' : ($atribuicao['status_mt'] == 'N/A' ? 'N/A' : 'Pendente')) ?>"><?= $atribuicao['status_mt'] ?></span>
                                    </td>
                                    
                                    <td class="px-6 py-4 whitespace-nowrap text-center text-sm font-medium">
                                        <a href="historico_lancamentos.php?professor_id=<?= $atribuicao['professor_id'] ?>&nome_professor=<?= urlencode($atribuicao['nome_professor']) ?>&trimestre=<?= $trimestre_ativo ?>"
                                           class="text-indigo-600 hover:text-indigo-900 flex items-center justify-center gap-1">
                                            <i class="fas fa-file-invoice"></i> Histórico
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </section>
</main>
</body>
</html>