<?php
session_start();
// O arquivo 'conexao.php' deve conter a conexão $conn = new mysqli(...)
include 'conexao.php'; 

// Adiciona esta linha para garantir a codificação UTF-8
if (isset($conn)) {
    $conn->set_charset("utf8");
}

// 1. Verificação de sessão e permissões
if (!isset($_SESSION['usuario']) || !in_array(strtoupper($_SESSION['usuario']['perfil']), ['SUBDIRECÇÃO PEDAGÓGICA', 'ADMINISTRADOR'])) {
    header("Location: login.php");
    exit;
}

$perfil_gestor = strtoupper($_SESSION['usuario']['perfil']);
$ano_atual = date('Y');

// 2. Coleta e validação dos parâmetros
$professor_id = filter_input(INPUT_GET, 'professor_id', FILTER_VALIDATE_INT);
$nome_professor = filter_input(INPUT_GET, 'nome_professor', FILTER_SANITIZE_STRING);

// Lógica de Seleção de Trimestre (Pode ser passado via GET)
$trimestre_ativo = filter_input(INPUT_GET, 'trimestre', FILTER_VALIDATE_INT);
if (!$trimestre_ativo || $trimestre_ativo < 1 || $trimestre_ativo > 3) {
    $trimestre_ativo = 1; // Padrão
}

if (!$professor_id) {
    echo "<p class='p-4 bg-red-100 text-red-700'>Erro: Professor não especificado.</p>";
    exit;
}

// Array para armazenar os dados organizados
$dados_historico = [];

// 3. Consulta de Atribuições (Turma/Disciplina) do Professor
$query_atribuicoes = "
    SELECT 
        atr.turma_id,
        atr.disciplina_id,
        t.classe,
        t.nome AS nome_turma,
        t.curso,
        d.nome AS nome_disciplina
    FROM atribuicoes atr
    JOIN turmas t ON atr.turma_id = t.id
    JOIN disciplinas d ON atr.disciplina_id = d.id
    WHERE atr.professor_id = ? 
    ORDER BY t.classe, t.nome, d.nome
";

$stmt_atr = $conn->prepare($query_atribuicoes);
$stmt_atr->bind_param("i", $professor_id);
$stmt_atr->execute();
$result_atribuicoes = $stmt_atr->get_result();

while ($atribuicao = $result_atribuicoes->fetch_assoc()) {
    $turma_id = $atribuicao['turma_id'];
    $disciplina_id = $atribuicao['disciplina_id'];

    // 4. Query Detalhada de Alunos e Notas
    $query_detalhes_notas = "
        SELECT
            a.nome AS nome_aluno,
            n.mac,
            n.npp,
            n.npt,
            n.mt,
            a.observacoes -- Traz a observação da tabela ALUNOS
        FROM matriculas m
        JOIN alunos a ON m.aluno_id = a.id
        LEFT JOIN notas n ON n.aluno_id = m.aluno_id
                         AND n.turma_id = m.turma_id
                         AND n.disciplina_id = ?
                         AND n.trimestre = ?
        WHERE m.turma_id = ?
          AND m.ano_letivo = ?
          -- CRITÉRIO DEFINITIVO: Inclui TODOS, exceto se a.observacoes contiver 'Desistente'
          -- O LIKE é case-insensitive por padrão no MySQL para a maioria dos collations, 
          -- mas adicionamos a verificação de NULL/vazio para cobrir todos os casos.
          AND (a.observacoes IS NULL 
               OR TRIM(a.observacoes) = '' 
               OR a.observacoes NOT LIKE '%Desistente%') 
        ORDER BY a.nome ASC
    ";

    $stmt_notas = $conn->prepare($query_detalhes_notas);
    // Tipos: i (disciplina_id), i (trimestre_ativo), i (turma_id), s (ano_atual)
    $stmt_notas->bind_param("iiis", $disciplina_id, $trimestre_ativo, $turma_id, $ano_atual);
    
    if (!$stmt_notas->execute()) {
        die("Erro ao executar a consulta de notas: " . $stmt_notas->error);
    }
    
    $result_notas = $stmt_notas->get_result();
    
    $alunos_notas = [];
    while ($nota = $result_notas->fetch_assoc()) {
        $alunos_notas[] = $nota;
    }
    $stmt_notas->close();
    
    // Armazena os dados organizados por atribuição
    $atribuicao['alunos_notas'] = $alunos_notas;
    $dados_historico[] = $atribuicao;
}

$stmt_atr->close();
$conn->close();

/**
 * Função utilitária para formatar notas (MAC, NPP, NPT, MT)
 */
function formatar_nota($nota) {
    if (is_null($nota) || $nota === '') {
        return '-'; // Nota não lançada
    }
    // Formata a nota para ter no máximo 2 casas decimais e substitui ponto por vírgula
    return number_format($nota, 2, ',', '');
}
?>
<!DOCTYPE html>
<html lang="pt">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Histórico de Lançamentos de <?= htmlspecialchars($nome_professor) ?></title>
<link rel="stylesheet" href="https://rsms.me/inter/inter.css">
<script src="https://cdn.tailwindcss.com"></script>
<link rel="stylesheet" href="css/all.min.css">
<style>
    body { font-family: 'Inter', sans-serif; }
    .nota-lancada { color: #1f2937; font-weight: 600; }
    .nota-pendente { color: #dc2626; font-weight: 500; }
</style>
</head>
<body class="bg-gray-100 flex min-h-screen">

<aside class="w-60 bg-[#0d1b2a] text-white flex flex-col">
    <div class="p-4 text-center border-b border-gray-700">
        <h1 class="text-xl font-bold text-blue-400">SISMAG</h1>
        <p class="text-xs text-gray-400 mt-1"><?= $perfil_gestor ?></p>
    </div>

    <nav class="flex-1 mt-3">
        <a href="painel_gestor.php" class="block px-5 py-3 hover:bg-[#1b263b] transition">
            <i class="fas fa-home text-blue-400 mr-2"></i> Início
        </a>
        <a href="historico_controlo.php" class="block px-5 py-3 bg-[#1b263b] transition">
            <i class="fas fa-arrow-left text-blue-400 mr-2"></i> Voltar ao Controlo
        </a>
        <a href="logout.php" class="block px-5 py-3 hover:bg-red-700 transition">
            <i class="fas fa-sign-out-alt text-red-400 mr-2"></i> Sair
        </a>
    </nav>
</aside>

<main class="flex-1 flex flex-col">
    <header class="bg-blue-700 text-white p-4 shadow flex justify-between items-center">
        <h2 class="text-lg font-semibold flex items-center gap-2">
            <i class="fas fa-file-invoice"></i> Histórico Detalhado de Lançamentos
        </h2>
    </header>

    <section class="flex-1 p-6">
        <div class="bg-white rounded-xl shadow-lg p-6">
            
            <form method="GET" class="mb-6 flex items-end gap-4 border-b pb-4">
                <input type="hidden" name="professor_id" value="<?= $professor_id ?>">
                <input type="hidden" name="nome_professor" value="<?= htmlspecialchars($nome_professor) ?>">
                <div class="flex-grow max-w-xs">
                    <label for="trimestre" class="block text-sm font-medium text-gray-700 mb-1">Visualizar Trimestre</label>
                    <select id="trimestre" name="trimestre" class="mt-1 block w-full pl-3 pr-10 py-2 text-base border-gray-300 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm rounded-md">
                        <option value="1" <?= $trimestre_ativo == 1 ? 'selected' : '' ?>>1º Trimestre</option>
                        <option value="2" <?= $trimestre_ativo == 2 ? 'selected' : '' ?>>2º Trimestre</option>
                        <option value="3" <?= $trimestre_ativo == 3 ? 'selected' : '' ?>>3º Trimestre</option>
                    </select>
                </div>
                <button type="submit" class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 transition duration-150">
                    <i class="fas fa-sync-alt mr-2"></i> Atualizar
                </button>
            </form>

            <h3 class="text-2xl font-bold text-gray-800 mb-4">
                Professor: <span class="text-indigo-600"><?= htmlspecialchars($nome_professor) ?></span>
            </h3>
            <p class="text-gray-600 mb-6">Visualizando detalhes do <span class="font-semibold text-blue-500"><?= $trimestre_ativo ?>º Trimestre</span>.</p>

            <?php if (empty($dados_historico)): ?>
                <div class="p-4 bg-yellow-100 border-l-4 border-yellow-500 text-yellow-700">
                    <p class="font-bold">Aviso:</p>
                    <p>Nenhuma atribuição de turma/disciplina encontrada para este professor neste ano letivo.</p>
                </div>
            <?php else: ?>
                <?php foreach ($dados_historico as $item): ?>
                    <div class="mb-8 border border-gray-200 rounded-lg overflow-hidden shadow-md">
                        <div class="bg-gray-100 p-4 border-b border-gray-200">
                            <h4 class="text-lg font-semibold text-gray-700">
                                <i class="fas fa-users mr-2 text-blue-500"></i> Turma: **<?= htmlspecialchars($item['classe']) . 'ª-' . htmlspecialchars($item['nome_turma']) ?>** (<?= htmlspecialchars($item['curso']) ?>)
                            </h4>
                            <p class="text-sm text-gray-500">
                                <i class="fas fa-book mr-1 text-blue-500"></i> Disciplina: **<?= htmlspecialchars($item['nome_disciplina']) ?>**
                            </p>
                        </div>

                        <div class="overflow-x-auto">
                            <table class="min-w-full divide-y divide-gray-200">
                                <thead class="bg-gray-50">
                                    <tr>
                                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Aluno</th>
                                        <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">MAC</th>
                                        <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">NPP</th>
                                        <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">NPT</th>
                                        <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">MT</th>
                                    </tr>
                                </thead>
                                <tbody class="bg-white divide-y divide-gray-200">
                                    <?php if (empty($item['alunos_notas'])): ?>
                                        <tr>
                                            <td colspan="5" class="px-6 py-4 text-center text-sm text-gray-500">
                                                Nenhum aluno encontrado ou ativo nesta turma/disciplina.
                                            </td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($item['alunos_notas'] as $aluno_nota): ?>
                                            <tr class="hover:bg-indigo-50/20">
                                                <td class="px-6 py-3 whitespace-nowrap text-sm font-medium text-gray-900">
                                                    <?= htmlspecialchars($aluno_nota['nome_aluno']) ?>
                                                </td>
                                                <td class="px-6 py-3 whitespace-nowrap text-sm text-center <?= !is_null($aluno_nota['mac']) ? 'nota-lancada' : 'nota-pendente' ?>">
                                                    <?= formatar_nota($aluno_nota['mac']) ?>
                                                </td>
                                                <td class="px-6 py-3 whitespace-nowrap text-sm text-center <?= !is_null($aluno_nota['npp']) ? 'nota-lancada' : 'nota-pendente' ?>">
                                                    <?= formatar_nota($aluno_nota['npp']) ?>
                                                </td>
                                                <td class="px-6 py-3 whitespace-nowrap text-sm text-center <?= !is_null($aluno_nota['npt']) ? 'nota-lancada' : 'nota-pendente' ?>">
                                                    <?= formatar_nota($aluno_nota['npt']) ?>
                                                </td>
                                                <td class="px-6 py-3 whitespace-nowrap text-sm text-center <?= !is_null($aluno_nota['mt']) ? 'nota-lancada' : 'nota-pendente' ?>">
                                                    <?= formatar_nota($aluno_nota['mt']) ?>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </section>
</main>
</body>
</html>