// js/inatividade.js
// Timeout de inatividade para o SISMAG
// Redireciona para logout.php após TEMPO_INATIVIDADE milissegundos.
// Sincroniza logout entre abas via localStorage (chave: 'sismag_logout').

(function () {
    // Tempo de inatividade em milissegundos (15000 = 15 segundos)
    const TEMPO_INATIVIDADE = 30000;

    // Se true, exibe um alert antes de redirecionar. Se preferir logout silencioso, deixe false.
    const MOSTRAR_ALERT = true;

    let temporizador = null;
    let encerrado = false;

    function encerrarSessaoLocal() {
        if (encerrado) return;
        encerrado = true;

        try {
            // Marca em localStorage para notificar outras abas
            localStorage.setItem('sismag_logout', Date.now());
        } catch (e) {
            // ignore se localStorage não estiver disponível
        }

        if (MOSTRAR_ALERT) {
            // Mostra aviso (opcional). Depois redireciona.
            try { alert('Sessão encerrada por inatividade.'); } catch (e) { /* ignore */ }
        }

        // Redireciona para o ficheiro PHP que limpa a sessão (logout.php)
        window.location.href = 'logout.php';
    }

    function resetarTemporizador() {
        if (encerrado) return;
        clearTimeout(temporizador);
        temporizador = setTimeout(encerrarSessaoLocal, TEMPO_INATIVIDADE);
    }

    // Eventos que representam atividade do usuário
    const eventosAtividade = [
        'mousemove', 'mousedown', 'keydown', 'touchstart', 'click', 'scroll'
    ];

    eventosAtividade.forEach(evt => {
        document.addEventListener(evt, resetarTemporizador, { passive: true });
    });

    // Reinicia ao ganhar foco
    window.addEventListener('focus', resetarTemporizador);

    // Observa mudança de visibilidade (ex.: usuário volta à aba)
    document.addEventListener('visibilitychange', function () {
        if (document.visibilityState === 'visible') {
            resetarTemporizador();
        }
    });

    // Sincronização entre abas: se outra aba escreveu a chave 'sismag_logout', desliga aqui também
    window.addEventListener('storage', function (e) {
        if (!e) return;
        if (e.key === 'sismag_logout') {
            // Outra aba iniciou logout — encerra sem tentar escrever novamente
            if (!encerrado) {
                encerrado = true;
                // opcional: pequena mensagem antes de redirecionar
                if (MOSTRAR_ALERT) {
                    try { alert('Sessão encerrada por inatividade noutra aba.'); } catch (err) {}
                }
                window.location.href = 'logout.php';
            }
        }
    });

    // Inicializa o temporizador quando o script carrega
    resetarTemporizador();

    // Caso queira expirar a sessão também se o token localStorage for limpo
    // (por ex. admin invalidou sessão), pode ouvir outra chave customizada.
})();
