<?php
/**
 * MAPA DE APROVEITAMENTO ESCOLAR
 * Exibe as médias trimestrais (MT1, MT2 ou MT3) dos alunos de uma turma.
 */
session_start();
include 'conexao.php';

// =================================================================
// 1. AUTENTICAÇÃO E CONFIGURAÇÃO
// =================================================================
$perfil = strtoupper($_SESSION['usuario']['perfil'] ?? '');
if (!in_array($perfil, ['SUBDIRECÇÃO PEDAGÓGICA', 'SECRETARIA PEDAGÓGICA'])) {
    header("Location: login.php");
    exit;
}

setlocale(LC_TIME, 'pt_PT.UTF-8', 'pt_BR.UTF-8', 'portuguese');
date_default_timezone_set('Africa/Luanda');

// =================================================================
// 2. BUSCA DE DADOS INICIAIS
// =================================================================
$turmas = [];
$res = $conn->query("SELECT id, nome, classe, curso, turno, ano FROM turmas ORDER BY classe, nome");
if ($res) {
    while ($row = $res->fetch_assoc()) {
        $turmas[] = $row;
    }
    $res->free();
}

$turma_id = filter_input(INPUT_GET, 'turma_id', FILTER_VALIDATE_INT);
$trimestre = filter_input(INPUT_GET, 'trimestre', FILTER_VALIDATE_INT);

$alunos = $disciplinas = $dados = [];
$nome_turma = $classe_turma = $curso_turma = $trimestre_extenso = '';

// =================================================================
// 3. BUSCA DE INFORMAÇÕES DA TURMA
// =================================================================
if ($turma_id) {
    $stmt_turma = $conn->prepare("SELECT nome, classe, curso FROM turmas WHERE id = ?");
    $stmt_turma->bind_param("i", $turma_id);
    $stmt_turma->execute();
    $res_turma = $stmt_turma->get_result();
    if ($row_turma = $res_turma->fetch_assoc()) {
        $nome_turma = htmlspecialchars($row_turma['nome']);
        $classe_turma = htmlspecialchars($row_turma['classe']);
        $curso_turma = htmlspecialchars($row_turma['curso']);
    }
    $stmt_turma->close();
}

if ($trimestre == 1) $trimestre_extenso = 'I Trimestre';
elseif ($trimestre == 2) $trimestre_extenso = 'II Trimestre';
elseif ($trimestre == 3) $trimestre_extenso = 'III Trimestre';

// =================================================================
// 4. BUSCA DE ALUNOS, DISCIPLINAS E NOTAS
// =================================================================
if ($turma_id && $trimestre) {
    $stmt_alunos = $conn->prepare("SELECT id, nome FROM alunos WHERE turma_id = ? ORDER BY nome");
    $stmt_alunos->bind_param("i", $turma_id);
    $stmt_alunos->execute();
    $alunos = $stmt_alunos->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt_alunos->close();

    $stmt_disciplinas = $conn->prepare("SELECT id, nome FROM disciplinas WHERE turma_id = ?");
    $stmt_disciplinas->bind_param("i", $turma_id);
    $stmt_disciplinas->execute();
    $disciplinas = $stmt_disciplinas->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt_disciplinas->close();

    $mt_coluna = "mt" . $trimestre;
    $stmt_notas = $conn->prepare("SELECT {$mt_coluna} FROM notas WHERE aluno_id = ? AND disciplina_id = ?");

    foreach ($alunos as $i => $aluno) {
        $linha = ['n' => $i + 1, 'nome' => $aluno['nome'], 'notas' => [], 'obs' => ''];
        $todas_maiores_que_10 = true;

        foreach ($disciplinas as $disc) {
            $stmt_notas->bind_param("ii", $aluno['id'], $disc['id']);
            $stmt_notas->execute();
            $notas = $stmt_notas->get_result()->fetch_assoc();

            $media = round(floatval($notas[$mt_coluna] ?? 0), 1);
            $linha['notas'][$disc['id']] = $media;

            if ($media < 10) {



                $todas_maiores_que_10 = false;
            }
        }

        $linha['obs'] = $todas_maiores_que_10 ? 'Positivo' : 'Negativo';
        $dados[] = $linha;
    }
    $stmt_notas->close();
}

// =================================================================
// 5. FUNÇÃO DE ORDENAÇÃO DE DISCIPLINAS (CORRIGIDA)
// =================================================================
function ordenar_disciplinas($disciplinas) {
    // AGRUPAMENTO PADRÃO TRADICIONAL
    $grupos = [
        'LÍNGUAS' => [
            'LÍNGUA PORTUGUESA', 'L.PORTUGUESA', 'PORTUGUESA', 'LITERATURA', 'INGLÊS', 'FRANCÊS'
        ],
        'EXATAS' => [
            'MATEMÁTICA', 'INFORMÁTICA'
        ],
        'FÍSICAS E NATURAIS' => [
            'FÍSICA', 'QUÍMICA', 'BIOLOGIA'
        ],
        'SOCIAIS E HUMANAS' => [
            'HISTÓRIA', 'GEOGRAFIA', 'EDUCAÇÃO MORAL', 'EDUCAÇÃO CÍVICA', 'FILOSOFIA', 'SOCIOLOGIA'
        ],
        'OUTRAS' => [
            'EDUCAÇÃO FÍSICA', 'EDUCAÇÃO LABORAL', 'EDUCAÇÃO VISUAL', 'EDUCAÇÃO PLÁSTICA', 'EMPREENDEDORISMO', 'ARTES', 'MÚSICA'
        ]
    ];

    // Normalizar nome e identificar grupo
    foreach ($disciplinas as &$d) {
        $nome = mb_strtoupper(trim($d['nome']), 'UTF-8');
        $nome = preg_replace('/\s+/', ' ', $nome);
        $nome = str_replace(['.', 'Ê', 'Ã', 'Á', 'É', 'Í', 'Ó', 'Ú'], '', $nome);
        $d['nome_norm'] = $nome;
        $d['grupo'] = 'OUTRAS';
        foreach ($grupos as $grupo => $lista) {
            foreach ($lista as $padrao) {
                if (str_contains($nome, strtoupper($padrao))) {
                    $d['grupo'] = $grupo;
                    break 2;
                }
            }
        }
    }
    unset($d);

    // ORDEM DE EXIBIÇÃO FINAL
    $ordem_grupos = ['LÍNGUAS', 'EXATAS', 'FÍSICAS E NATURAIS', 'SOCIAIS E HUMANAS', 'OUTRAS'];

    // Ordenar primeiro pelos grupos e depois alfabeticamente dentro de cada grupo
    usort($disciplinas, function ($a, $b) use ($ordem_grupos) {
        $pos_a = array_search($a['grupo'], $ordem_grupos);
        $pos_b = array_search($b['grupo'], $ordem_grupos);
        if ($pos_a === $pos_b) {
            return strcmp($a['nome_norm'], $b['nome_norm']);
        }
        return $pos_a <=> $pos_b;
    });

    return $disciplinas;
}



// Aplica ordenação
$disciplinas = ordenar_disciplinas($disciplinas);
$data_atual = date('d/m/Y');
?>
<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mapa de Aproveitamento</title>
    <style>
        /* === ESTILOS GERAIS === */
        body { font-family: 'Segoe UI', sans-serif; margin: 0; background-color: #f4f7f9; color: #333; display: flex; justify-content: center; flex-direction: column; align-items: center; padding: 10px; box-sizing: border-box; }
        .container { max-width: 95%; width: 100%; background-color: #ffffff; padding: 15px; box-shadow: 0 2px 7px rgba(0, 0, 0, 0.1); border-radius: 5px; overflow: hidden; position: relative; }
        
        /* === FORMULÁRIO E CABEÇALHO === */
        .header-form-container { display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; margin-bottom: 10px; gap: 10px; flex-direction: row-reverse; /* Visto Director à Esquerda, Formulário à Direita */ }
        form { display: flex; align-items: center; flex-wrap: wrap; gap: 8px; }
        select, button { padding: 6px 10px; border: 1px solid #ccc; border-radius: 3px; font-size: 12px; transition: all 0.3s ease; }
        button { background-color: #007bff; color: white; cursor: pointer; border: none; }
        button:hover { background-color: #0056b3; }
        #print-button { background-color: #6c757d; }
        #print-button:hover { background-color: #5a6268; }

        .cabecalho { text-align: center; margin-bottom: 15px; line-height: 1.1; }
        .cabecalho img { width: 60px; height: auto; margin-bottom: 5px; }
        h2, h3 { margin: 0; font-weight: 600; font-size: 1.2em; }
        h3 { font-size: 0.9em; font-weight: 400; }
        .info-trimestre-turma { text-align: center; margin-bottom: 10px; font-size: 0.9em; font-weight: bold; }
        
        /* === VISTO DO DIRETOR (Para o cabeçalho) === */
        .visto-director { text-align: center; font-size: 0.8em; position: relative; }
        .visto-director .line { border-bottom: 1px dotted #000; width: 150px; margin: 5px auto; }
        .visto-director .director-nome { margin-top: 5px; font-weight: bold; }

        /* === TABELA === */
        .table-container { overflow-x: auto; width: 100%; }
        table { border-collapse: collapse; width: 100%; font-size: 10px; table-layout: fixed; }
        th, td { border: 1px solid #e0e0e0; padding: 5px; text-align: center; height: 30px; }
        thead th { background-color: #003366; color: white; font-weight: 600; vertical-align: middle; }
        tbody tr:nth-child(even) { background-color: #f9f9f9; }
        tbody tr:hover { background-color: #f1f1f1; }
        td.vermelho { color: #d9534f; font-weight: bold; }
        
        /* === LARGURA DE COLUNAS (TELA) === */
        <?php 
            $num_disciplinas = max(1, count($disciplinas));
            // Cálculo: 4%(Nº) + 15%(Nome) + 6%(Obs) = 25% fixo. 75% restante para disciplinas.
            $largura_disciplina_percent = 75 / $num_disciplinas; 
            $largura_disciplina_percent = min(10, max(3, $largura_disciplina_percent)); // Limite p/ legibilidade
            $largura_disciplina_string = number_format($largura_disciplina_percent, 2) . '%';
        ?>
        th:nth-child(1), td:nth-child(1) { width: 4%; } /* Nº */
        th:nth-child(2), td:nth-child(2) { width: 15%; text-align: left; word-break: break-word; } /* Nome */
        th.disciplina, td.disciplina { width: <?= $largura_disciplina_string ?>; word-break: break-word; white-space: normal; } /* Disciplina */
        th:last-child, td:last-child { width: 6%; } /* Obs. */

        /* === RODAPÉ: ASSINATURAS E DATA === */
        .assinaturas { width: 100%; display: flex; justify-content: space-between; margin-top: 25px; font-size: 0.9em; }
        .assinatura-esquerda, .assinatura-direita { text-align: center; width: 45%; }
        .assinatura-esquerda .line, .assinatura-direita .line { padding-top: 50px; border-top: 1px dashed #000; }
        .data-mapa { text-align: right; margin-top: 20px; font-size: 0.9em; }

       /* ============================================ */
/* ESTILOS PARA IMPRESSÃO A4 VERTICAL (PORTRAIT)*/
/* ============================================ */
@media print {
    @page { 
        size: A4 portrait; /* Vertical */
        margin: 8mm; /* Margens pequenas */
    }

    body { 
        background-color: #fff; 
        padding: 0; 
        zoom: 85%; /* Reduz um pouco para caber mais linhas */
        -webkit-print-color-adjust: exact !important;
        color-adjust: exact !important;
        font-family: "Arial", sans-serif;
    }

    .header-form-container, #print-button, button { 
        display: none !important; 
    } /* Oculta botões e formulários */

    .container { 
        box-shadow: none; 
        padding: 0; 
        max-width: 100%; 
        font-size: 8px; 
    }

    /* Cabeçalho */
    .cabecalho { margin-bottom: 5px; }
    .cabecalho img { width: 35px; }
    h2 { font-size: 0.9em; }
    h3 { font-size: 0.6em; }
    .info-trimestre-turma { font-size: 0.7em; margin-bottom: 5px; }

    /* === TABELA PRINCIPAL === */
    table { 
        width: 100%; 
        border-collapse: collapse; 
        font-size: 7px; 
        table-layout: fixed; 
    }

    th, td { 
        border: 1px solid #000; 
        padding: 2px 1px; 
        height: 10px; 
        white-space: normal !important; 
        text-align: center;
        word-wrap: break-word;
    }

    /* === AJUSTES DE LARGURA DE IMPRESSÃO === */
    <?php 
        // Cálculo: 3%(Nº) + 15%(Nome) + 5%(Obs) = 23% fixo. 77% restante dividido pelo nº de disciplinas.
        $fixo_print_percent = 3 + 15 + 5; 
        $livre_print_percent = 100 - $fixo_print_percent; 
        $largura_disciplina_print = $livre_print_percent / $num_disciplinas;
        $largura_disciplina_print_string = number_format($largura_disciplina_print, 2) . '%';
    ?>
    th:nth-child(1), td:nth-child(1) { width: 3%; } /* Nº */
    th:nth-child(2), td:nth-child(2) { width: 15%; text-align: left; padding-left: 2px; } /* Nome */
    th.disciplina, td.disciplina { width: <?= $largura_disciplina_print_string ?>; font-size: 6px; line-height: 1; }
    th:last-child, td:last-child { width: 5%; } /* Obs */

    /* === ASSINATURAS E RODAPÉ === */
    .assinaturas { margin-top: 10px; font-size: 0.6em; }
    .assinatura-esquerda .line, .assinatura-direita .line { padding-top: 25px; }
    .data-mapa { font-size: 0.6em; margin-top: 5px; }
    .visto-director { font-size: 0.6em; }
    .visto-director .line { width: 120px; }

    /* === LIMITADOR DE ALUNOS POR PÁGINA === */
    tbody tr:nth-of-type(41) { page-break-before: always; } 
}

    </style>
</head>
<body>

<div class="container">
    <div class="header-form-container">
        <div class="visto-director">
            O DIRECTOR DO COMPLEXO ESCOLAR
            <div class="line"></div>
            <div class="director-nome">Eduardo Sanjambela Tchandja</div>
        </div>

        <form method="get">
            <label style="font-size: 0.9em;">Turma:</label>
            <select name="turma_id" required style="font-size: 0.9em;">
                <option value="">-- Escolher Turma --</option>
                <?php foreach ($turmas as $t): ?>
                    <option value="<?= $t['id'] ?>" <?= ($t['id'] == $turma_id ? 'selected' : '') ?>>
                        <?= htmlspecialchars("{$t['classe']} - {$t['nome']} - {$t['curso']}") ?>
                    </option>
                <?php endforeach; ?>
            </select>

            <label style="font-size: 0.9em;">Trimestre:</label>
            <select name="trimestre" required style="font-size: 0.9em;">
                <option value="1" <?= $trimestre == 1 ? 'selected' : '' ?>>I</option>
                <option value="2" <?= $trimestre == 2 ? 'selected' : '' ?>>II</option>
                <option value="3" <?= $trimestre == 3 ? 'selected' : '' ?>>III</option>
            </select>

            <button type="submit" style="font-size: 0.9em;">Ver Mapa</button>
            <button id="print-button" type="button" onclick="window.print()" style="font-size: 0.9em;">Imprimir</button>
        </form>
    </div>

    <?php if ($turma_id && $trimestre): ?>
        <?php if (empty($alunos)): ?>
            <div style="text-align: center; margin-top: 20px; font-size: 1.1em; color: #cc3300;">
                Nenhum aluno encontrado ou notas não lançadas para os filtros selecionados.
            </div>
        <?php else: ?>
            <div class="cabecalho">
                <img src="insignia/insignia.png" alt="Insígnia da República de Angola">
                <h2>REPÚBLICA DE ANGOLA</h2>
                <h3>MINISTÉRIO DA EDUCAÇÃO</h3>
                <h3>MAPA DE APROVEITAMENTO ESCOLAR</h3>
            </div>

            <div class="info-trimestre-turma">
                <?= htmlspecialchars($trimestre_extenso) ?> - Turma: <?= htmlspecialchars($classe_turma) ?> - <?= htmlspecialchars($nome_turma) ?> - <?= htmlspecialchars($curso_turma) ?>
            </div>

            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th rowspan="2">Nº</th>
                            <th rowspan="2">Nome do Aluno</th>
                            <th colspan="<?= count($disciplinas) ?>">Disciplinas</th>
                            <th rowspan="2">Obs.</th>
                        </tr>
                        <tr>
                            <?php foreach ($disciplinas as $d): ?>
                                <th class="disciplina"><?= htmlspecialchars($d['nome']) ?></th>
                            <?php endforeach; ?>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($dados as $linha): ?>
                            <tr>
                                <td><?= $linha['n'] ?></td>
                                <td><?= htmlspecialchars($linha['nome']) ?></td>
                                <?php foreach ($disciplinas as $d):
                                    $nota = $linha['notas'][$d['id']] ?? 0;
                                    $display_nota = ($nota === 0.0 || $nota === 0) ? '--' : $nota;
                                ?>
                                    <td class="<?= is_numeric($nota) && $nota > 0 && $nota < 10 ? 'vermelho' : '' ?>">
                                        <?= $display_nota ?>
                                    </td>
                                <?php endforeach; ?>
                                <td class="<?= $linha['obs'] === 'Negativo' ? 'vermelho' : '' ?>"><?= $linha['obs'] ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <div class="data-mapa">
                Catumbela, aos <?= $data_atual ?>
            </div>

            <div class="assinaturas">
                <div class="assinatura-esquerda">
                    <p>Assinatura do Director de Turma</p>
                    <div class="line"></div>
                </div>
                <div class="assinatura-direita">
                    <p>Assinatura do Subdirecto Pedagógico</p>
                    <div class="line"></div>
                </div>
            </div>
        <?php endif; ?>
    <?php endif; ?>
</div>


<script>
document.addEventListener('DOMContentLoaded', function() {
    const linhas = document.querySelectorAll('table tbody tr');
    const maxLinhas = 40;

    if (linhas.length > maxLinhas) {
        const tabela = document.querySelector('table');
        const novaTabela = tabela.cloneNode(true);
        const corpoNovo = novaTabela.querySelector('tbody');
        corpoNovo.innerHTML = '';

        for (let i = maxLinhas; i < linhas.length; i++) {
            corpoNovo.appendChild(linhas[i].cloneNode(true));
            linhas[i].remove();
        }

        const pageBreak = document.createElement('div');
        pageBreak.style.pageBreakBefore = 'always';
        document.body.appendChild(pageBreak);
        document.body.appendChild(novaTabela);
    }
});
</script>




</body>
</html>