<?php
session_start();
include 'conexao.php'; // Certifique-se de que 'conexao.php' está correto

// 1. VERIFICAÇÃO DE PERMISSÃO: Permite o acesso se for Professor ou se estiver simulando (Gestor).
$usuario_id = (int)($_SESSION['usuario']['id'] ?? 0);
$perfil = $_SESSION['usuario']['perfil'] ?? '';
$is_professor = strtoupper(trim($perfil)) === 'PROFESSOR';
$is_simulating = $_SESSION['simulando'] ?? false; 

// Redireciona se não tiver permissão
if (!$is_professor && !$is_simulating) {
    header('Location: login.php');
    exit;
}

// O ID do professor cujas turmas serão listadas (professor alvo na simulação)
$professor_id_alvo = $usuario_id; 

// 2. RECEBER PARÂMETROS DE FILTRO
$turma_id = $_GET['turma_id'] ?? null;
$disciplina_id = $_GET['disciplina_id'] ?? null;

if (!$turma_id || !$disciplina_id || !is_numeric($turma_id) || !is_numeric($disciplina_id)) {
    die("Parâmetros de Turma e Disciplina são obrigatórios para esta pauta.");
}

// 3. BUSCAR DADOS DE TURMA, DISCIPLINA E PROFESSOR (Para o Cabeçalho)
$nome_professor = '';
$nome_disciplina = '';
$dados_turma = [];

// Nome do Professor
$sqlNome = "SELECT nome FROM usuarios WHERE id = ? LIMIT 1";
$stmt = $conn->prepare($sqlNome);
if ($stmt) {
    $stmt->bind_param("i", $professor_id_alvo);
    $stmt->execute();
    $resNome = $stmt->get_result();
    if ($resNome->num_rows === 1) {
        $nome_professor = $resNome->fetch_assoc()['nome'];
    }
    $stmt->close();
}


// Nome da Disciplina
$sqlDisc = "SELECT nome FROM disciplinas WHERE id = ?";
$stmt = $conn->prepare($sqlDisc);
if ($stmt) {
    $stmt->bind_param("i", $disciplina_id);
    $stmt->execute();
    $resDisc = $stmt->get_result();
    if ($resDisc->num_rows === 1) {
        $nome_disciplina = $resDisc->fetch_assoc()['nome'];
    }
    $stmt->close();
}


// Dados da Turma
$sqlTurma = "SELECT nome, classe, sala, curso, turno FROM turmas WHERE id = ?";
$stmt = $conn->prepare($sqlTurma);
if ($stmt) {
    $stmt->bind_param("i", $turma_id);
    $stmt->execute();
    $resTurma = $stmt->get_result();
    if ($resTurma->num_rows === 1) {
        $dados_turma = $resTurma->fetch_assoc();
    }
    $stmt->close();
}


// 4. BUSCAR ALUNOS E TODAS AS NOTAS (1º, 2º e 3º Trimestre)
$alunos = [];
$sqlAlunos = "
    SELECT 
        a.id AS aluno_id, 
        a.nome AS aluno_nome,
        -- NOTAS DO 1º TRIMESTRE (MAC|NPP|NPT|MT)
        (SELECT CONCAT(IFNULL(n.mac, ''), '|', IFNULL(n.npp, ''), '|', IFNULL(n.npt, ''), '|', IFNULL(n.mt, '')) FROM notas n WHERE n.aluno_id = a.id AND n.disciplina_id = ? AND n.turma_id = ? AND n.trimestre = 1) AS notas_t1,
        -- NOTAS DO 2º TRIMESTRE (MAC|NPP|NPT|MT)
        (SELECT CONCAT(IFNULL(n.mac, ''), '|', IFNULL(n.npp, ''), '|', IFNULL(n.npt, ''), '|', IFNULL(n.mt, '')) FROM notas n WHERE n.aluno_id = a.id AND n.disciplina_id = ? AND n.turma_id = ? AND n.trimestre = 2) AS notas_t2,
        -- NOTAS DO 3º TRIMESTRE + FINAL (MAC|NPP|NPT|MT|MFD|MF)
        (SELECT CONCAT(IFNULL(n.mac, ''), '|', IFNULL(n.npp, ''), '|', IFNULL(n.npt, ''), '|', IFNULL(n.mt, ''), '|', IFNULL(n.mfd, ''), '|', IFNULL(n.mf, '')) FROM notas n WHERE n.aluno_id = a.id AND n.disciplina_id = ? AND n.turma_id = ? AND n.trimestre = 3) AS notas_t3_final
    FROM 
        alunos a
    WHERE 
        a.turma_id = ?
    ORDER BY 
        a.nome
";

$stmt = $conn->prepare($sqlAlunos);

// Total de 7 parâmetros (o mesmo número, pois as novas colunas estão dentro da mesma tabela 'notas')
if ($stmt) {
    $stmt->bind_param("iiiiiii", 
        $disciplina_id, $turma_id, // T1 (2)
        $disciplina_id, $turma_id, // T2 (2)
        $disciplina_id, $turma_id, // T3 (2)
        $turma_id // Turma principal (1)
    );
    $stmt->execute();
    $result = $stmt->get_result();

    while ($row = $result->fetch_assoc()) {
        // Processamento das notas (separação pelo delimitador '|')
        
        // T1: MAC|NPP|NPT|MT
        $t1_parts = explode('|', $row['notas_t1'] ?? '|||');
        $row['t1_mac'] = $t1_parts[0] ?? '';
        $row['t1_npp'] = $t1_parts[1] ?? ''; // NOVA COLUNA
        $row['t1_npt'] = $t1_parts[2] ?? '';
        $row['t1_mt']  = $t1_parts[3] ?? '';

        // T2: MAC|NPP|NPT|MT
        $t2_parts = explode('|', $row['notas_t2'] ?? '|||');
        $row['t2_mac'] = $t2_parts[0] ?? '';
        $row['t2_npp'] = $t2_parts[1] ?? ''; // NOVA COLUNA
        $row['t2_npt'] = $t2_parts[2] ?? '';
        $row['t2_mt']  = $t2_parts[3] ?? '';

        // T3: MAC|NPP|NPT|MT|MFD|MF (6 partes)
        $t3_parts = explode('|', $row['notas_t3_final'] ?? '|||||');
        $row['t3_mac'] = $t3_parts[0] ?? '';
        $row['t3_npp'] = $t3_parts[1] ?? ''; // NOVA COLUNA
        $row['t3_npt'] = $t3_parts[2] ?? '';
        $row['t3_mt']  = $t3_parts[3] ?? '';
        $row['mfd']    = $t3_parts[4] ?? '';
        $row['mf']     = $t3_parts[5] ?? '';
        
        $alunos[] = $row;
    }
    $stmt->close();
}


$conn->close();

// Funções de formatação (permanecem as mesmas)
function formatar_nota($nota) {
    if ($nota === '' || $nota === null) return '';
    return number_format((float)$nota, 1, ',', '.'); 
}
function formatar_nota_final($nota) {
    if ($nota === '' || $nota === null) return '';
    return number_format((float)$nota, 0); // Sem casas decimais
}

// 5. VARIÁVEIS DO CABEÇALHO/RODAPÉ (Padrões)
$ano_lectivo_atual = '2025/2026';
$nome_republica = "REPÚBLICA DE ANGOLA";
$nome_governo = "GOVERNO PROVINCIAL DE BENGUELA";
$nome_gabinete = "GABINETE PROVINCIAL DA EDUCAÇÃO";
$nome_escola = "COMPLEXO ESCOLAR BG0026 PADRE EMIL FRICK - CATUMBELA";
$nome_sub_diretor = "Anacleto Domingos de Oliveira Cachiuenjo";
?>
<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mini-Pauta Anual de <?= htmlspecialchars($nome_disciplina) ?></title>
    
    <script src="js/tailwindcss.js"></script>
    <link rel="stylesheet" href="css/all.min.css">

    <style>
        body { margin: 0; padding: 0; background: white; }
        .print-only { display: block; }
        .screen-only { display: none; }
        
        .screen-only-container {
            padding: 20px; 
            background: #f0f0f0; 
            display: flex;
            flex-direction: column;
            align-items: center;
        }
        .preview-box {
            background: #fff;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            width: 100%;
            max-width: 700px;
            text-align: center;
        }

        /* ================================================= */
        /* ESTILOS DE IMPRESSÃO (A4 VERTICAL) - CRÍTICO */
        /* ================================================= */
        @media print {
            .screen-only { display: none !important; }

            @page {
                size: A4 portrait; 
                margin: 5mm; 
                padding: 0;
            }

            html, body {
                width: 210mm;
                height: 297mm;
                overflow: hidden;
            }

            .container-print {
                width: 100%;
                height: 100%;
                padding: 0;
                font-family: Arial, sans-serif;
            }

            /* --- CABEÇALHO --- */
            .header-print {
                text-align: center;
                margin-bottom: 5px;
            }
            .header-insignia img { width: 35px; height: auto; margin: 0 auto; display: block; } /* Reduzido levemente a insígnia */
            .header-text { line-height: 1.1; font-size: 7.5pt; margin-bottom: 3px; } /* Reduzido levemente o texto */
            .header-text p { margin: 0; }
            .header-title { font-weight: bold; margin: 3px 0; font-size: 8.5pt; } /* Reduzido levemente o título */
            
            .header-info { 
                text-align: left; 
                font-size: 7pt; /* Muito pequeno para caber */
                width: 100%; 
                margin-top: 5px; 
                display: flex;
                flex-wrap: wrap;
                justify-content: space-between;
                padding: 0 5mm;
            }
            .header-info > div { width: 50%; box-sizing: border-box; }
            .header-info p { margin: 1px 0; }
            .hr-divisor { border: 0; border-top: 1px solid #000; margin: 5px 0; }
            
            /* --- TABELA PRINCIPAL (2+4+4+4+2 = 16 Colunas) --- */
            table {
                width: 100%;
                border-collapse: collapse;
                font-size: 6.5pt; /* Reduzido para caber 16 colunas em A4 vertical */
                table-layout: fixed;
            }

            th, td {
                border: 1px solid #000 !important;
                padding: 0.5px; /* Padding MÍNIMO */
                text-align: center;
                height: 14px; /* Altura da linha para caber 40 linhas */
                white-space: nowrap;
                overflow: hidden;
            }
            
            thead {
                -webkit-print-color-adjust: exact;
                color-adjust: exact;
                background-color: #d1d5db !important; 
                color: #000 !important;
            }
            
            /* Larguras das Colunas */
            .col-num { width: 3%; } /* 3% */
            .col-nome { width: 23%; text-align: left !important; padding-left: 3px !important; } /* 23% */
            
            /* Colunas de Notas (12 colunas no total para as notas parciais) */
            /* Cada nota (MAC/NPP/NPT/MT) deve ter 4.33% de largura (13*4.33% = 56.29%) */
            /* 4 notas * 3 trimestres = 12 colunas */
            
            /* Larguras Finais */
            .col-mfd { width: 3.5%; } /* 3.5% */
            .col-mf { width: 3.5%; } /* 3.5% */
            
            /* Ajustando as 12 colunas de notas para ocupar o espaço restante */
            td:nth-child(3), td:nth-child(4), td:nth-child(5), td:nth-child(6), /* T1 (4 colunas) */
            td:nth-child(7), td:nth-child(8), td:nth-child(9), td:nth-child(10), /* T2 (4 colunas) */
            td:nth-child(11), td:nth-child(12), td:nth-child(13), td:nth-child(14) /* T3 (4 colunas) */
            {
                /* 100% - 3% (Nº) - 23% (Nome) - 3.5% (MFD) - 3.5% (MF) = 67% */
                /* 67% / 12 colunas ≈ 5.58% por coluna */
                width: 5.58%; 
            }
        
            /* --- RODAPÉ --- */
            .footer-print {
                margin-top: 10px;
                width: 100%;
                font-size: 7pt; /* Reduzido */
                padding: 0 5mm;
            }
            .assinatura-grid {
                display: flex;
                justify-content: space-around;
                width: 95%; 
                margin: 0 auto;
                text-align: center;
            }
            .assinatura-col { width: 45%; }
            .linha-assinatura {
                margin-top: 10px; /* Reduzido o espaço */
                border-bottom: 1px solid #000;
                width: 100%;
                height: 1px;
            }
            .data-catumbela { margin-bottom: 3px; }
        }
        
        .screen-only { display: block; }
    </style>
</head>
<body>

<div class="screen-only screen-only-container">
    <div class="preview-box">
        <h1 class="text-xl font-bold text-gray-800 mb-2">Pré-Visualização da Caderneta Anual</h1>
        <p class="mb-4 text-gray-600">Documento Oficial: 3 Trimestres em 1 Folha A4 Vertical.</p>
        
        <button onclick="window.print()" class="bg-blue-700 text-white py-3 px-6 rounded-lg hover:bg-blue-800 transition shadow-md flex items-center justify-center mx-auto text-lg font-semibold">
            <i class="fas fa-print mr-3"></i> Gerar Impressão (Configurar Impressora)
        </button>
        
        <p class="mt-4 text-xs text-gray-500">
            **Ajuste Obrigatório:** Na janela de impressão, configure: <br>
            **Layout:** Vertical (Portrait) | **Papel:** A4 | **Escala:** Padrão (100%)
        </p>
    </div>
</div>

<div class="print-only container-print">

    <div class="header-print">
        <div class="header-insignia">
            <img src="insignia/insignia.png" alt="Emblema da República">
        </div>
        
        <div class="header-text">
            <p style="font-weight: bold;"><?= htmlspecialchars($nome_republica) ?></p>
            <p><?= htmlspecialchars($nome_governo) ?></p>
            <p><?= htmlspecialchars($nome_gabinete) ?></p>
            <p style="font-weight: bold;"><?= htmlspecialchars($nome_escola) ?></p>
        </div>

        <p class="header-title">MINI-PAUTA GERAL (CADERNETA ANUAL)</p>

        <div class="header-info">
            <div>
                <p>Disciplina: <span style="font-weight: bold;"><?= htmlspecialchars($nome_disciplina) ?></span></p>
                <p>Professor: <span style="font-weight: bold;"><?= htmlspecialchars($nome_professor) ?></span></p>
            </div>
            <div>
                <p>Classe: <span style="font-weight: bold;"><?= htmlspecialchars($dados_turma['classe'] ?? '') ?>º / Turma: <span style="font-weight: bold;"><?= htmlspecialchars($dados_turma['nome'] ?? '') ?></span></p>
                <p>Turno: <span style="font-weight: bold;"><?= htmlspecialchars($dados_turma['turno'] ?? '') ?> / Sala: <span style="font-weight: bold;"><?= htmlspecialchars($dados_turma['sala'] ?? '') ?></span></p>
            </div>
        </div>
        <div class="header-info" style="justify-content: center;">
             <p>Ano Lectivo: <span style="font-weight: bold;"><?= htmlspecialchars($ano_lectivo_atual) ?></span></p>
        </div>
        <div class="hr-divisor"></div>
    </div>
    
    <table border="1">
        <thead>
            <tr>
                <th rowspan="2" class="col-num">Nº</th>
                <th rowspan="2" class="col-nome">Nome do Aluno</th>
                
                <th colspan="4">1º TRIMESTRE</th> <th colspan="4">2º TRIMESTRE</th> <th colspan="4">3º TRIMESTRE</th> <th rowspan="2" class="col-mfd">MFD</th>
                <th rowspan="2" class="col-mf">MF</th>
            </tr>
            <tr>
                <th>MAC</th>
                <th>NPP</th> 
                <th>NPT</th>
                <th>MT</th>
                
                <th>MAC</th>
                <th>NPP</th>
                <th>NPT</th>
                <th>MT</th>
                
                <th>MAC</th>
                <th>NPP</th>
                <th>NPT</th>
                <th>MT</th>
            </tr>
        </thead>
        <tbody>
            <?php 
            $i = 1; 
            $max_alunos = 40;
            $alunos_count = count($alunos);

            foreach ($alunos as $aluno): ?>
                <tr>
                    <td class="col-num"><?= $i++ ?></td>
                    <td class="col-nome"><?= htmlspecialchars($aluno['aluno_nome']) ?></td>
                    
                    <td><?= formatar_nota($aluno['t1_mac']) ?></td>
                    <td><?= formatar_nota($aluno['t1_npp']) ?></td>
                    <td><?= formatar_nota($aluno['t1_npt']) ?></td>
                    <td><?= formatar_nota($aluno['t1_mt']) ?></td>
                    
                    <td><?= formatar_nota($aluno['t2_mac']) ?></td>
                    <td><?= formatar_nota($aluno['t2_npp']) ?></td>
                    <td><?= formatar_nota($aluno['t2_npt']) ?></td>
                    <td><?= formatar_nota($aluno['t2_mt']) ?></td>
                    
                    <td><?= formatar_nota($aluno['t3_mac']) ?></td>
                    <td><?= formatar_nota($aluno['t3_npp']) ?></td>
                    <td><?= formatar_nota($aluno['t3_npt']) ?></td>
                    <td><?= formatar_nota($aluno['t3_mt']) ?></td>
                    
                    <td class="col-mfd"><?= formatar_nota_final($aluno['mfd']) ?></td>
                    <td class="col-mf"><?= formatar_nota_final($aluno['mf']) ?></td>
                </tr>
            <?php endforeach; ?>
            
            <?php for ($j = $alunos_count; $j < $max_alunos; $j++): ?>
                <tr>
                    <td class="col-num"><?= $i++ ?></td>
                    <td class="col-nome"></td>
                    <td></td><td></td><td></td><td></td>
                    <td></td><td></td><td></td><td></td>
                    <td></td><td></td><td></td><td></td>
                    <td class="col-mfd"></td>
                    <td class="col-mf"></td>
                </tr>
            <?php endfor; ?>
        </tbody>
    </table>
    
    <div class="footer-print">
         <p class="data-catumbela" style="text-align: right;">A Catumbela, aos ____ de ________________ de <?= date('Y') ?>.</p>

        <div class="assinatura-grid">
            <div class="assinatura-col">
                <div class="linha-assinatura"></div>
                <p>O/A Professor(a):</p>
                <p style="font-weight: bold;"><?= htmlspecialchars($nome_professor) ?></p>
            </div>
            <div class="assinatura-col">
                <div class="linha-assinatura"></div>
                <p>O Subdirector Pedagógico:</p>
                <p style="font-weight: bold;"><?= htmlspecialchars($nome_sub_diretor) ?></p>
            </div>
        </div>
    </div>
</div>

</body>
</html>