<?php
session_start();
include 'conexao.php';

$usuario_id = (int)($_SESSION['usuario']['id'] ?? 0);
$perfil = $_SESSION['usuario']['perfil'] ?? '';

// Se não for professor ou não estiver logado, redireciona para login
if (strtoupper(trim($perfil)) !== 'PROFESSOR' || $usuario_id <= 0) {
    header('Location: login.php');
    exit;
}

// Carregar nome do professor
$nome_professor = '';
$sqlNome = "SELECT nome FROM usuarios WHERE id = ? LIMIT 1";
$stmt = $conn->prepare($sqlNome);
if ($stmt) {
    $stmt->bind_param("i", $usuario_id);
    $stmt->execute();
    $resNome = $stmt->get_result();
    if ($resNome && $resNome->num_rows === 1) {
        $rowNome = $resNome->fetch_assoc();
        $nome_professor = $rowNome['nome'];
    }
    $stmt->close();
}

// Inicialização de variáveis para filtros
$turma_id = isset($_GET['turma_id']) ? (int)$_GET['turma_id'] : null;
$disciplina_id = isset($_GET['disciplina_id']) ? (int)$_GET['disciplina_id'] : null;
$trimestre_selecionado = isset($_GET['trimestre']) ? (int)$_GET['trimestre'] : null;
$nome_disciplina = '';

// Carregar turmas para o professor
$turmas = false;
$sqlTurmas = "
    SELECT DISTINCT t.id, CONCAT(t.classe, ' - ', t.nome, ' - ', t.curso, ' - ', t.turno) AS descricao
    FROM turmas t
    INNER JOIN atribuicoes a ON t.id = a.turma_id
    WHERE a.professor_id = ?
      AND a.ano_lectivo = '2025/2026'
    ORDER BY t.classe, t.nome
";
$stmt = $conn->prepare($sqlTurmas);
if ($stmt) {
    $stmt->bind_param("i", $usuario_id);
    $stmt->execute();
    $turmas = $stmt->get_result();
    $stmt->close();
}

// Carregar disciplinas da turma selecionada para o professor
$disciplinas = false;
if ($turma_id) {
    $sqlDisciplinas = "
        SELECT DISTINCT d.id, d.nome
        FROM atribuicoes a
        INNER JOIN disciplinas d ON a.disciplina_id = d.id
        WHERE a.turma_id = ?
          AND a.professor_id = ?
          AND a.ano_lectivo = '2025/2026'
        ORDER BY d.nome
    ";
    $stmt = $conn->prepare($sqlDisciplinas);
    if ($stmt) {
        $stmt->bind_param("ii", $turma_id, $usuario_id);
        $stmt->execute();
        $disciplinas = $stmt->get_result();
        $stmt->close();
    }
}

// Obter o nome da disciplina selecionada
if ($disciplina_id) {
    $sqlNomeDisciplina = "SELECT nome FROM disciplinas WHERE id = ?";
    $stmt = $conn->prepare($sqlNomeDisciplina);
    if ($stmt) {
        $stmt->bind_param("i", $disciplina_id);
        $stmt->execute();
        $res = $stmt->get_result();
        if ($res->num_rows > 0) {
            $row = $res->fetch_assoc();
            $nome_disciplina = $row['nome'];
        }
        $stmt->close();
    }
}

// Carregar alunos e notas se todos os filtros estiverem selecionados
$alunos = [];
$notas_por_aluno = [];
if ($turma_id && $disciplina_id) {
    // Buscar todos os alunos da turma
    $sqlAlunos = "SELECT id, nome FROM alunos WHERE turma_id = ? ORDER BY nome";
    $stmtAlunos = $conn->prepare($sqlAlunos);
    $stmtAlunos->bind_param("i", $turma_id);
    $stmtAlunos->execute();
    $alunos_res = $stmtAlunos->get_result();
    $alunos_data = $alunos_res->fetch_all(MYSQLI_ASSOC);
    $stmtAlunos->close();
    
    // Agora, buscar TODAS as notas para todos os trimestres para esses alunos
    $aluno_ids = array_column($alunos_data, 'id');
    if (!empty($aluno_ids)) {
        $in_clause = str_repeat('?,', count($aluno_ids) - 1) . '?';
        $sqlNotas = "SELECT * FROM notas WHERE aluno_id IN ($in_clause) AND disciplina_id = ? AND turma_id = ?";
        
        $params_types = str_repeat('i', count($aluno_ids)) . 'ii';
        $params = array_merge($aluno_ids, [$disciplina_id, $turma_id]);
        
        $stmtNotas = $conn->prepare($sqlNotas);
        $stmtNotas->bind_param($params_types, ...$params);
        $stmtNotas->execute();
        $notas_res = $stmtNotas->get_result();
        
        while ($nota = $notas_res->fetch_assoc()) {
            $notas_por_aluno[$nota['aluno_id']][$nota['trimestre']] = $nota;
        }
        $stmtNotas->close();
    }
    $alunos = $alunos_data;
}
?>
<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <title>Mini Pauta Trimestres</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        body {
            font-family: 'Inter', sans-serif;
        }
        .nota-baixa {
            color: #ef4444; /* red-500 */
            font-weight: bold;
        }
        
        /* Estilos de impressão */
        @media print {
            @page {
                size: a4 landscape;
            }
            .no-print, .topo-form {
                display: none !important;
            }
            body {
                background-color: #ffffff !important;
                font-size: 10px !important;
                padding: 0 !important;
                margin: 0 !important; /* Adicionado para garantir que não haja margem na página impressa */
            }
            header {
                text-align: center;
                margin-bottom: 10px;
            }
            header img {
                width: 80px;
            }
            h1, h2 {
                font-size: 12px;
            }
            .painel {
                box-shadow: none !important;
                border-radius: 0 !important;
                padding: 0 !important;
                overflow-x: visible !important;
            }
            table {
                width: 100% !important;
                min-width: 100% !important;
                font-size: 10px !important;
                table-layout: fixed; /* Adicionado para forçar a tabela a se ajustar */
            }
            th, td {
                padding: 2px !important;
                word-wrap: break-word;
                white-space: normal;
                overflow: hidden; /* Evita que o conteúdo da célula saia */
                text-overflow: ellipsis; /* Adiciona '...' se o texto for muito longo */
            }
            input[type="number"][readonly] {
                border: none !important; /* Remove a borda dos campos readonly na impressão */
                background-color: transparent !important; /* Remove o fundo */
                padding: 0 !important;
            }
            .print-flex {
                display: flex !important;
                justify-content: space-around !important;
            }
            .signatures-section {
                margin-top: 0 !important; /* Remove o espaçamento entre a tabela e as assinaturas */
            }
            /* Remove a largura fixa dos inputs para se ajustarem à tabela na impressão */
            table input[type="number"] {
                width: auto !important; 
            }
        }
    </style>
    <script>
        document.addEventListener('DOMContentLoaded', () => {
            const form = document.querySelector('form[action="salvar_notas.php"]');
            const nomeDisciplina = form ? form.getAttribute('data-disciplina-nome').toLowerCase() : '';
            const trimestreSelecionado = parseInt(document.querySelector('select[name="trimestre"]').value) || 1;

            function pintarNotas(linha) {
                const notasCampos = linha.querySelectorAll(
                    `input[name^="av"][type="number"],
                     input[name^="npt"][type="number"],
                     input[name^="neo"][type="number"],
                     input[name^="nee"][type="number"],
                     .mt1_display, .mt2_display, .mt3_display, .mfd_display, .mf_display, .mec_display`
                );
                
                notasCampos.forEach(campo => {
                    let valorNota = parseFloat(campo.value || campo.textContent);
                    
                    if (isNaN(valorNota) || valorNota <= 1) {
                        campo.classList.remove('nota-baixa');
                        return;
                    }
                    
                    if (valorNota < 10) {
                        campo.classList.add('nota-baixa');
                    } else {
                        campo.classList.remove('nota-baixa');
                    }
                });
            }

            function calcularNotas(linha) {
                const alunoId = linha.getAttribute('data-aluno-id');

                // 1. Calcular a MT do trimestre atual
                const av1 = parseFloat(linha.querySelector(`input[name="av1_${trimestreSelecionado}_${alunoId}"]`)?.value) || 0;
                const av2 = parseFloat(linha.querySelector(`input[name="av2_${trimestreSelecionado}_${alunoId}"]`)?.value) || 0;
                const av3 = parseFloat(linha.querySelector(`input[name="av3_${trimestreSelecionado}_${alunoId}"]`)?.value) || 0;
                const npt = parseFloat(linha.querySelector(`input[name="npt_${trimestreSelecionado}_${alunoId}"]`)?.value) || 0;

                const mtAtual = (av1 + av2 + av3 + npt) / 4;
            
                // Atualiza a MT na tela e no campo oculto
                linha.querySelector(`.mt${trimestreSelecionado}_display`).textContent = mtAtual.toFixed(1);
                linha.querySelector(`input[name="mt${trimestreSelecionado}_${alunoId}"]`).value = mtAtual.toFixed(2);

                // 2. Coletar as notas de todos os trimestres
                const mt1 = parseFloat(linha.querySelector(`input[name="mt1_${alunoId}"]`)?.value) || 0;
                const mt2 = parseFloat(linha.querySelector(`input[name="mt2_${alunoId}"]`)?.value) || 0;
                const mt3 = parseFloat(linha.querySelector(`input[name="mt3_${alunoId}"]`)?.value) || 0;
            
                // 3. Calcular MFD
                const mfd = (mt1 + mt2 + mt3) / 3;
                linha.querySelector('.mfd_display').textContent = mfd.toFixed(1);
                linha.querySelector(`input[name="mfd_${alunoId}"]`).value = mfd.toFixed(2);

                // 4. Calcular MEC e MF (se for o 3º trimestre ou houver dados de NEO/NEE)
                const neo = parseFloat(linha.querySelector(`input[name="neo_${alunoId}"]`)?.value) || 0;
                const nee = parseFloat(linha.querySelector(`input[name="nee_${alunoId}"]`)?.value) || 0;
            
                let mf = 0;
                let mec = 0;

                if (neo > 0 || nee > 0) {
                    if (nomeDisciplina.includes('portuguesa') || nomeDisciplina.includes('inglesa')) {
                        mec = (neo + nee) / 2;
                    } else {
                        mec = nee;
                    }
                    
                    linha.querySelector('.mec_display').textContent = mec.toFixed(1);
                    linha.querySelector(`input[name="mec_${alunoId}"]`).value = mec.toFixed(2);
                    mf = (mfd * 0.6 + mec * 0.4);
                } else {
                    linha.querySelector('.mec_display').textContent = '0.0';
                    linha.querySelector(`input[name="mec_${alunoId}"]`).value = 0;
                    mf = mfd;
                }
            
                linha.querySelector('.mf_display').textContent = Math.round(mf);
                linha.querySelector(`input[name="mf_${alunoId}"]`).value = Math.round(mf);
            
                pintarNotas(linha);
            }
            
            document.querySelectorAll('tbody tr').forEach(linha => {
                linha.querySelectorAll('input[type="number"]').forEach(input => {
                    if (!input.readOnly) {
                        input.addEventListener('input', () => calcularNotas(linha));
                    }
                });
                calcularNotas(linha);
            });
        });
    </script>
</head>
<body class="bg-white text-gray-800 p-4 sm:p-8">

<div class="no-print fixed top-4 right-4 z-50 flex space-x-2">
    <button type="submit" form="notas-form" class="bg-blue-600 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded-lg shadow-lg transition-colors">Salvar</button>
    <button type="button" onclick="window.print()" class="bg-blue-600 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded-lg shadow-lg transition-colors">Imprimir</button>
    <button type="button" onclick="window.location.href = 'painel_professor.php'" class="bg-gray-500 hover:bg-gray-600 text-white font-bold py-2 px-4 rounded-lg shadow-lg transition-colors">Voltar</button>
</div>

<div class="no-print absolute top-8 left-8 text-sm text-gray-600 font-medium">
    MINI PAUTAS DO(A) PROFESSOR(A): <strong class="text-gray-900"><?= htmlspecialchars($nome_professor ?? '') ?></strong>
</div>

<div class="max-w-7xl mx-auto bg-white rounded-xl shadow-lg p-6 sm:p-10">
    <header class="text-center mb-8">
        <img src="insignia/insignia.png" alt="Insígnia" class="w-24 mx-auto mb-4">
        <h1 class="text-xl sm:text-2xl font-bold text-gray-900">REPÚBLICA DE ANGOLA</h1>
        <h2 class="text-md sm:text-lg font-semibold text-gray-700">GOVERNO PROVINCIAL DE BENGUELA</h2>
        <h2 class="text-md sm:text-lg font-semibold text-gray-700">GABINETE PROVINCIAL DA EDUCAÇÃO</h2>
        <h2 class="text-md sm:text-lg font-semibold text-gray-700">COMPLEXO ESCOLAR BG0026 PADRE EMIL FRICK – CATUMBELA</h2>
        <h2 class="text-xl sm:text-2xl font-bold text-blue-800 mt-4">MINI-PAUTA</h2>
    </header>

    <div class="topo-form mb-6 p-4 rounded-lg shadow-sm flex flex-wrap items-center justify-center space-y-4 md:space-y-0 md:space-x-4">
        <form method="GET" action="" class="w-full md:w-auto flex flex-wrap items-center justify-center space-y-4 md:space-y-0 md:space-x-4">
            <div class="flex items-center space-x-2">
                <label for="turma_id" class="font-medium text-gray-700">Turma:</label>
                <select name="turma_id" id="turma_id" class="form-select border-gray-300 rounded-lg shadow-sm focus:border-blue-500 focus:ring focus:ring-blue-500 focus:ring-opacity-50">
                    <option value="">- Selecione a Turma -</option>
                    <?php
                    if ($turmas) {
                        while($row = $turmas->fetch_assoc()) {
                            $selected = ($row['id'] == $turma_id) ? 'selected' : '';
                            echo "<option value=\"{$row['id']}\" {$selected}>" . htmlspecialchars($row['descricao']) . "</option>";
                        }
                    }
                    ?>
                </select>
            </div>
            
            <div class="flex items-center space-x-2">
                <label for="disciplina_id" class="font-medium text-gray-700">Disciplina:</label>
                <select name="disciplina_id" id="disciplina_id" class="form-select border-gray-300 rounded-lg shadow-sm focus:border-blue-500 focus:ring focus:ring-blue-500 focus:ring-opacity-50">
                    <option value="">- Selecione a Disciplina -</option>
                    <?php
                    if ($disciplinas) {
                        while($row = $disciplinas->fetch_assoc()) {
                            $selected = ($row['id'] == $disciplina_id) ? 'selected' : '';
                            echo "<option value=\"{$row['id']}\" {$selected}>" . htmlspecialchars($row['nome']) . "</option>";
                        }
                    }
                    ?>
                </select>
            </div>
            
            <div class="flex items-center space-x-2">
                <label for="trimestre" class="font-medium text-gray-700">Trimestre:</label>
                <select name="trimestre" id="trimestre" class="form-select border-gray-300 rounded-lg shadow-sm focus:border-blue-500 focus:ring focus:ring-blue-500 focus:ring-opacity-50">
                    <option value="">- Selecione o Trimestre -</option>
                    <?php for($i=1; $i<=3; $i++): ?>
                        <option value="<?= $i ?>" <?= ($i == $trimestre_selecionado) ? 'selected' : '' ?>>
                            Trimestre <?= $i ?>
                        </option>
                    <?php endfor; ?>
                </select>
            </div>
            
            <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded-lg shadow-md transition-colors">Ver Pauta</button>
        </form>
    </div>
    
    <?php if (!empty($alunos) && $disciplina_id && $trimestre_selecionado): ?>
    <form id="notas-form" method="POST" action="salvar_notas.php" data-disciplina-nome="<?= htmlspecialchars($nome_disciplina) ?>">
        <input type="hidden" name="disciplina_id" value="<?= htmlspecialchars($disciplina_id) ?>">
        <input type="hidden" name="turma_id" value="<?= htmlspecialchars($turma_id) ?>">
        <input type="hidden" name="trimestre" value="<?= htmlspecialchars($trimestre_selecionado) ?>">
        <p class="mb-4 text-center font-bold">
            <span class="text-gray-700">Disciplina:</span> <span class="text-blue-800"><?= htmlspecialchars($nome_disciplina) ?></span>, 
            <span class="text-gray-700">Trimestre:</span> <span class="text-blue-800"><?= htmlspecialchars($trimestre_selecionado) ?></span>
        </p>

        <div class="overflow-x-auto rounded-lg shadow-md">
            <table class="w-full text-sm text-left text-gray-600">
                <thead class="text-xs text-gray-700 uppercase bg-blue-100">
                    <tr>
                        <th scope="col" class="py-3 px-2 border-r border-blue-200" rowspan="2">Nº</th>
                        <th scope="col" class="py-3 px-2 border-r border-blue-200" rowspan="2">Nome</th>
                        <?php for($t=1; $t<=3; $t++): ?>
                            <th scope="col" class="py-3 px-2 text-center border-r border-blue-200" colspan="5">Trimestre <?= $t ?></th>
                        <?php endfor; ?>
                        <th scope="col" class="py-3 px-2 text-center border-r border-blue-200" rowspan="2">MFD</th>
                        <th scope="col" class="py-3 px-2 text-center border-r border-blue-200" rowspan="2">NEO</th>
                        <th scope="col" class="py-3 px-2 text-center border-r border-blue-200" rowspan="2">NEE</th>
                        <th scope="col" class="py-3 px-2 text-center border-r border-blue-200" rowspan="2">MEC</th>
                        <th scope="col" class="py-3 px-2 text-center" rowspan="2">MF</th>
                    </tr>
                    <tr>
                        <?php for($t=1; $t<=3; $t++): ?>
                            <th scope="col" class="py-2 px-1 text-center border-r border-blue-200">AV1</th>
                            <th scope="col" class="py-2 px-1 text-center border-r border-blue-200">AV2</th>
                            <th scope="col" class="py-2 px-1 text-center border-r border-blue-200">AV3</th>
                            <th scope="col" class="py-2 px-1 text-center border-r border-blue-200">NPT</th>
                            <th scope="col" class="py-2 px-1 text-center border-r border-blue-200">MT<?= $t ?></th>
                        <?php endfor; ?>
                    </tr>
                </thead>
                <tbody>
                <?php $n=1; foreach($alunos as $aluno):
                    $aluno_id = $aluno['id'];
                ?>
                <tr class="bg-white border-b hover:bg-gray-50" data-aluno-id="<?= htmlspecialchars($aluno_id) ?>">
                    <td class="py-2 px-2 border-r"><?= $n++ ?></td>
                    <td class="py-2 px-2 border-r whitespace-nowrap">
                        <?= htmlspecialchars($aluno['nome']) ?>
                        <input type="hidden" name="aluno_id[]" value="<?= htmlspecialchars($aluno_id) ?>">
                    </td>
                    
                    <?php for ($t = 1; $t <= 3; $t++):
                        $notas = $notas_por_aluno[$aluno_id][$t] ?? [];
                    ?>
                        <td class="py-2 px-1 border-r">
                            <input type="number" name="av1_<?= $t ?>_<?= $aluno_id ?>" value="<?= htmlspecialchars($notas['av1'] ?? '') ?>" min="0" max="20" step="0.1" <?= $t != $trimestre_selecionado ? 'readonly' : '' ?> class="w-12 text-center p-1 border border-gray-300 rounded-md focus:border-blue-500 focus:ring focus:ring-blue-500 focus:ring-opacity-50">
                        </td>
                        <td class="py-2 px-1 border-r">
                            <input type="number" name="av2_<?= $t ?>_<?= $aluno_id ?>" value="<?= htmlspecialchars($notas['av2'] ?? '') ?>" min="0" max="20" step="0.1" <?= $t != $trimestre_selecionado ? 'readonly' : '' ?> class="w-12 text-center p-1 border border-gray-300 rounded-md focus:border-blue-500 focus:ring focus:ring-blue-500 focus:ring-opacity-50">
                        </td>
                        <td class="py-2 px-1 border-r">
                            <input type="number" name="av3_<?= $t ?>_<?= $aluno_id ?>" value="<?= htmlspecialchars($notas['av3'] ?? '') ?>" min="0" max="20" step="0.1" <?= $t != $trimestre_selecionado ? 'readonly' : '' ?> class="w-12 text-center p-1 border border-gray-300 rounded-md focus:border-blue-500 focus:ring focus:ring-blue-500 focus:ring-opacity-50">
                        </td>
                        <td class="py-2 px-1 border-r">
                            <input type="number" name="npt_<?= $t ?>_<?= $aluno_id ?>" value="<?= htmlspecialchars($notas['npt'] ?? '') ?>" min="0" max="20" step="0.1" <?= $t != $trimestre_selecionado ? 'readonly' : '' ?> class="w-12 text-center p-1 border border-gray-300 rounded-md focus:border-blue-500 focus:ring focus:ring-blue-500 focus:ring-opacity-50">
                        </td>
                        <td class="py-2 px-1 border-r font-semibold">
                            <span class="mt<?= $t ?>_display"><?= htmlspecialchars($notas["mt{$t}"] ?? '0.0') ?></span>
                            <input type="hidden" name="mt<?= $t ?>_<?= $aluno_id ?>" value="<?= htmlspecialchars($notas["mt{$t}"] ?? '0') ?>">
                        </td>
                    <?php endfor; ?>
                    
                    <?php $notas_finais = $notas_por_aluno[$aluno_id][3] ?? []; ?>
                    <td class="py-2 px-1 border-r font-bold">
                        <span class="mfd_display"><?= htmlspecialchars($notas_finais['mfd'] ?? '0.0') ?></span>
                        <input type="hidden" name="mfd_<?= $aluno_id ?>" value="<?= htmlspecialchars($notas_finais['mfd'] ?? '0') ?>">
                    </td>
                    <td class="py-2 px-1 border-r">
                        <input type="number" name="neo_<?= $aluno_id ?>" value="<?= htmlspecialchars($notas_finais['neo'] ?? '') ?>" min="0" max="20" step="0.1" <?= $trimestre_selecionado != 3 || (!stristr($nome_disciplina, 'portuguesa') && !stristr($nome_disciplina, 'inglesa')) ? 'readonly' : '' ?> class="w-12 text-center p-1 border border-gray-300 rounded-md focus:border-blue-500 focus:ring focus:ring-blue-500 focus:ring-opacity-50">
                    </td>
                    <td class="py-2 px-1 border-r">
                        <input type="number" name="nee_<?= $aluno_id ?>" value="<?= htmlspecialchars($notas_finais['nee'] ?? '') ?>" min="0" max="20" step="0.1" <?= $trimestre_selecionado != 3 ? 'readonly' : '' ?> class="w-12 text-center p-1 border border-gray-300 rounded-md focus:border-blue-500 focus:ring focus:ring-blue-500 focus:ring-opacity-50">
                    </td>
                    <td class="py-2 px-1 border-r font-bold">
                        <span class="mec_display"><?= htmlspecialchars($notas_finais['mec'] ?? '0.0') ?></span>
                        <input type="hidden" name="mec_<?= $aluno_id ?>" value="<?= htmlspecialchars($notas_finais['mec'] ?? '0') ?>">
                    </td>
                    <td class="py-2 px-1 font-bold">
                        <span class="mf_display"><?= htmlspecialchars($notas_finais['mf'] ?? '0') ?></span>
                        <input type="hidden" name="mf_<?= $aluno_id ?>" value="<?= htmlspecialchars($notas_finais['mf'] ?? '0') ?>">
                    </td>
                </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </form>
    <?php else: ?>
        <p class="text-center text-lg text-gray-500 py-8">Por favor, selecione uma turma, disciplina e trimestre para visualizar a pauta.</p>
    <?php endif; ?>
</div>

<div class="max-w-7xl mx-auto mb-4 text-center">
    <div class="grid grid-cols-1 md:grid-cols-2 print-flex gap-8 signatures-section">
        <div>
            <p class="font-bold text-gray-800 mb-2">O(A) PROFESSOR(A)</p>
            <div class="border-b-2 border-gray-400 mt-12 w-3/4 mx-auto"></div>
            <p class="mt-2 text-sm text-gray-600"><?= htmlspecialchars($nome_professor ?? '') ?></p>
            <p class="mt-4 text-sm text-gray-500">CATUMBELA, AOS <?= date("d/m/Y"); ?></p>
        </div>
        <div>
            <p class="font-bold text-gray-800 mb-2">O(A) SUB-DIRECTOR(A) PEDAGÓGICO(A)</p>
            <div class="border-b-2 border-gray-400 mt-12 w-3/4 mx-auto"></div>
            <p class="mt-2 text-sm text-gray-600">Anacleto Domingos de Oliveira Cachiyenjo</p>
            <p class="mt-4 text-sm text-gray-500">CATUMBELA, AOS <?= date("d/m/Y"); ?></p>
        </div>
    </div>
</div>

<?php
$conn->close();
?>

</body>
</html>