<?php
session_start();
include 'conexao.php';

// Define os perfis de usuário que têm permissão para acessar este painel.
$perfis_permitidos = ['ADMINISTRADOR'];

// Verifica se o usuário está logado e se o perfil dele está na lista de perfis permitidos.
if (!isset($_SESSION['usuario']) || !in_array(strtoupper($_SESSION['usuario']['perfil']), $perfis_permitidos)) {
    header("Location: login.php");
    exit;
}

$usuario = $_SESSION['usuario'];
$nome = htmlspecialchars($usuario['nome'] ?? 'Funcionário(a)');
$foto = htmlspecialchars($usuario['foto'] ?? 'sem-foto.png');
$mensagem_upload = '';

// Limpa a mensagem da sessão para que ela não seja exibida novamente após a atualização da página
if (isset($_SESSION['mensagem_upload'])) {
    $mensagem_upload = $_SESSION['mensagem_upload'];
    unset($_SESSION['mensagem_upload']);
}

// LÓGICA PARA PROCESSAR O UPLOAD DA FOTO DE PERFIL
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['upload_foto'])) {
    if (isset($_FILES['nova_foto']) && $_FILES['nova_foto']['error'] === UPLOAD_ERR_OK) {
        $diretorio_destino = 'fotos/';

        // Apenas para fins de demonstração, verifica a extensão
        // Em um ambiente de produção, é recomendável uma validação mais robusta
        $extensao = pathinfo($_FILES['nova_foto']['name'], PATHINFO_EXTENSION);
        $extensoes_permitidas = ['jpeg', 'jpg', 'png', 'gif'];
        if (!in_array(strtolower($extensao), $extensoes_permitidas)) {
            $_SESSION['mensagem_upload'] = "❌ Erro: Tipo de arquivo não permitido. Apenas JPG, PNG e GIF são aceitos.";
            header("Location: painel_diretor_geral.php");
            exit;
        }

        // Garante que o diretório de destino existe
        if (!is_dir($diretorio_destino)) {
            mkdir($diretorio_destino, 0755, true);
        }

        $foto_nova_nome = uniqid('perfil_') . '.' . $extensao;
        $caminho_completo = $diretorio_destino . $foto_nova_nome;

        if (move_uploaded_file($_FILES['nova_foto']['tmp_name'], $caminho_completo)) {
            // Atualiza o nome da foto no banco de dados
            $usuario_id = $_SESSION['usuario']['id'];
            $stmt = $conn->prepare("UPDATE usuarios SET foto = ? WHERE id = ?");
            $stmt->bind_param("si", $foto_nova_nome, $usuario_id);
            if ($stmt->execute()) {
                // Remove a foto antiga se não for a padrão para evitar acúmulo de arquivos
                if (isset($_SESSION['usuario']['foto']) && $_SESSION['usuario']['foto'] !== 'sem-foto.png') {
                    $caminho_antigo = $diretorio_destino . $_SESSION['usuario']['foto'];
                    if (file_exists($caminho_antigo)) {
                        unlink($caminho_antigo);
                    }
                }
                
                // Atualiza a variável de sessão para que a foto seja exibida imediatamente
                $_SESSION['usuario']['foto'] = $foto_nova_nome;
                $foto = $foto_nova_nome;
                $_SESSION['mensagem_upload'] = "✅ Foto de perfil atualizada com sucesso!";
            } else {
                $_SESSION['mensagem_upload'] = "❌ Erro ao atualizar o banco de dados.";
                // Remova a foto do servidor se o banco de dados falhar
                unlink($caminho_completo);
            }
            $stmt->close();
        } else {
            $_SESSION['mensagem_upload'] = "❌ Erro ao fazer o upload da foto.";
        }
    } else {
        $_SESSION['mensagem_upload'] = "⚠️ Por favor, selecione um arquivo de foto.";
    }
    // Redireciona de volta para o painel para exibir a mensagem e evitar reenvio do formulário
    header("Location: painel_diretor_geral.php");
    exit();
}

// Consultas para os cartões de estatística (Total de Alunos, Usuários, Turmas).
$qtdAlunos = 0; $qtdUsuarios = 0; $qtdTurmas = 0;
if ($res = $conn->query("SELECT COUNT(*) AS total FROM alunos")) $qtdAlunos = $res->fetch_assoc()['total'];
if ($res = $conn->query("SELECT COUNT(*) AS total FROM usuarios")) $qtdUsuarios = $res->fetch_assoc()['total'];
if ($qtdTurmasRes = $conn->query("SELECT COUNT(*) AS total FROM turmas")) $qtdTurmas = $qtdTurmasRes->fetch_assoc()['total'];

// --- CONSULTAS DINÂMICAS PARA OS GRÁFICOS ---

// 1. Gráfico de Pizza: Distribuição de Alunos por Classe
$query_alunos_por_turma = "SELECT t.nome as nome_turma, COUNT(a.id) as total_alunos
FROM turmas t
LEFT JOIN alunos a ON t.id = a.turma_id
GROUP BY t.id";
$result_alunos_por_turma = $conn->query($query_alunos_por_turma);

$labels_classe = [];
$data_classe = [];
if ($result_alunos_por_turma->num_rows > 0) {
    while ($row = $result_alunos_por_turma->fetch_assoc()) {
        $labels_classe[] = $row['nome_turma'];
        $data_classe[] = (int)$row['total_alunos'];
    }
} else {
    $labels_classe = ['Sem turmas'];
    $data_classe = [0];
}

// 2. Gráfico de Barras: Evolução de Notas por Classe
$query_notas_por_trimestre = "SELECT t.nome as nome_turma,
AVG(n.mt1) as media_mt1,
AVG(n.mt2) as media_mt2,
AVG(n.mt3) as media_mt3
FROM notas n
JOIN alunos a ON n.aluno_id = a.id
JOIN turmas t ON a.turma_id = t.id
GROUP BY t.nome";
$result_notas = $conn->query($query_notas_por_trimestre);

$labels_notas = ['1º Trimestre', '2º Trimestre', '3º Trimestre'];
$datasets_notas = [];
if ($result_notas->num_rows > 0) {
    $colors = ['#007bff', '#28a745', '#ffc107', '#dc3545', '#6f42c1', '#17a2b8', '#fd7e14', '#e83e8c'];
    $color_index = 0;
    while ($row = $result_notas->fetch_assoc()) {
        $datasets_notas[] = [
            'label' => $row['nome_turma'],
            'data' => [(float)$row['media_mt1'], (float)$row['media_mt2'], (float)$row['media_mt3']],
            'backgroundColor' => $colors[$color_index % count($colors)],
            'borderColor' => $colors[$color_index % count($colors)],
            'borderWidth' => 1
        ];
        $color_index++;
    }
} else {
    $datasets_notas = [
        ['label' => 'Sem dados', 'data' => [0, 0, 0], 'backgroundColor' => '#ccc']
    ];
}

$conn->close();
?>

<!DOCTYPE html>

<html lang="pt">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<script src="js/chart.umd.js"></script>
<title>DIRECTOR GERAL</title>
<style>
/* Estilos reset e globais */
* { margin: 0; padding: 0; box-sizing: border-box; }
body {
    font-family: 'Segoe UI', sans-serif;
    background: #e9ebee;
    display: flex;
    height: 100vh;
    overflow: hidden;
}

/* --- Estilos do topo (Top Bar) --- */
.topbar {
    position: fixed;
    top: 0;
    left: 250px;
    right: 0;
    height: 60px;
    background: #003366; /* Azul Escuro */
    color: white;
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 0 20px;
    z-index: 1000;
    transition: left 0.3s;
}
.user-info {
    display: flex;
    align-items: center;
    gap: 10px;
}
.user-info img {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    object-fit: cover;
    border: 2px solid white;
    cursor: pointer; /* Adicionado cursor pointer */
}
.menu-toggle {
    display: none;
    cursor: pointer;
    font-size: 24px;
    margin-right: 15px;
}

/* --- Estilos da Barra Lateral (Sidebar) --- */
.sidebar {
    width: 250px;
    background: #003366; /* Azul Escuro */
    color: white;
    overflow-y: auto;
    padding-top: 60px;
    position: fixed;
    left: 0;
    top: 0;
    bottom: 0;
    transition: transform 0.3s ease-in-out;
    z-index: 999;
}
.sidebar.active {
    transform: translateX(0);
}
.sidebar .menu-item {
    padding: 15px 20px;
    cursor: pointer;
    font-weight: bold;
    transition: background 0.3s;
    display: flex;
    align-items: center;
    gap: 10px;
}
.sidebar .menu-item:hover {
    background: rgba(255,255,255,0.1);
}
.sidebar .submenu-sidebar {
    padding-left: 20px;
}
.sidebar .submenu-sidebar a {
    color: #fff;
    display: block;
    padding: 8px 0;
    text-decoration: none;
}

/* --- Estilos do Conteúdo Principal (Main Content) --- */
.main-content {
    margin-top: 60px;
    margin-left: 250px;
    padding: 30px;
    width: calc(100% - 250px);
    height: calc(100vh - 60px);
    overflow-y: auto;
    transition: margin-left 0.3s;
}
.statistics-container {
    display: flex;
    flex-wrap: wrap;
    gap: 20px;
    margin-bottom: 30px;
}
.stat-card {
    background: white;
    padding: 20px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    flex: 1;
    min-width: 200px;
    text-align: center;
    transition: transform 0.2s;
}
.stat-card:hover {
    transform: translateY(-5px);
}
.stat-card h3 {
    color: #0d1b2a;
    margin-bottom: 10px;
}
.stat-card p {
    font-size: 1.8em;
    font-weight: bold;
    color: #333;
}
.graficos-container {
    display: flex;
    gap: 30px;
    flex-wrap: wrap;
    justify-content: space-between;
    margin-bottom: 30px;
}
.grafico-bloco {
    background: white;
    border-radius: 10px;
    padding: 20px;
    flex: 1;
    min-width: 280px;
    max-width: 600px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}
.grafico-bloco canvas {
    max-width: 100%;
    height: auto;
}

/* Estilo da mensagem de sucesso/erro */
.mensagem {
    margin-top: 10px;
    font-weight: bold;
    padding: 10px;
    border-radius: 5px;
    text-align: center;
}
.mensagem.sucesso {
    color: #28a745;
    background-color: #d4edda;
    border: 1px solid #c3e6cb;
}
.mensagem.erro {
    color: #dc3545;
    background-color: #f8d7da;
    border: 1px solid #f5c6cb;
}

/* --- Media Queries para Responsividade --- */
@media (max-width: 768px) {
    .menu-toggle { display: block; }
    .sidebar { transform: translateX(-250px); }
    .sidebar.active { transform: translateX(0); }
    .topbar {
        left: 0;
        width: 100%;
        justify-content: flex-start;
        gap: 20px;
    }
    .main-content { margin-left: 0; width: 100%; }
    .topbar .topbar-left { display: none; }
}
</style>
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.2/dist/chart.umd.min.js"></script>
</head>
<body>

<div class="topbar">
    <div class="menu-toggle" onclick="toggleMenu()">
        <span>☰</span>
    </div>
    <div class="topbar-left">
        <strong>PAINEL DO DIRECTOR GERAL</strong> | Bem-vindo, <?= htmlspecialchars($nome) ?>
    </div>
    <div class="user-info">
        <form action="" method="post" enctype="multipart/form-data" id="fotoUploadForm">
            <label for="nova_foto">
                <img src="fotos/<?= htmlspecialchars($foto) ?>?t=<?= time() ?>" alt="Foto" id="userPhoto">
            </label>
            <input type="file" name="nova_foto" id="nova_foto" style="display:none;" onchange="document.getElementById('fotoUploadForm').submit();">
            <input type="hidden" name="upload_foto" value="1">
        </form>
        <span><?= htmlspecialchars($nome) ?></span>
    </div>
</div>

<div class="sidebar" id="sidebar">
    <div class="menu-item">📊 Painel Principal</div>
    <div class="menu-item">➕ Cadastramento</div>
    <div class="submenu-sidebar">
        <a href="cadastrar_aluno1.php">Cadastrar Aluno</a>
        <a href="cadastrar_usuario.php">Cadastrar Usuário</a>
        <a href="cadastrar_turma1.php">Cadastrar Turma</a>
        <a href="documentos_recebidos.php">Documentos Recebidos</a>
        <a href="enviar_documento.php">Enviar Documento</a>
    </div>
    <div class="menu-item">⚙️ Gestão</div>
    <div class="submenu-sidebar">
        <a href="relatorios.php">Relatórios</a>
        <a href="estatistica.php">Estatísticas</a>
    </div>
    <div class="menu-item">🛡️ <a href="painel_subadmin1.php" style="color:white; text-decoration:none;">Subdir. Administrativa</a></div>
    <div class="menu-item">🎓 <a href="painel_subpedagogico1.php" style="color:white; text-decoration:none;">Subdir. Pedagógica</a></div>
    <div class="menu-item">🧑‍🏫 <a href="cadastrar_aviso_professor.php" style="color:white; text-decoration:none;">Informações aos Professores</a></div>
    <div class="menu-item">🏢 <a href="painel_secretaria_geral1.php" style="color:white; text-decoration:none;">Secretaria Geral</a></div>
    <div class="menu-item">📖 <a href="painel_secpedagogica1.php" style="color:white; text-decoration:none;">Secretaria Pedagógica</a></div>
    <div class="menu-item">👥 <a href="painel_professor1.php" style="color:white; text-decoration:none;">Professores</a></div>
    <div class="menu-item">📢 <a href="cadastrar_aviso.php" style="color:white; text-decoration:none;">Informações aos alunos</a></div>
    <div class="menu-item">⚙️ <a href="configuracoes.php" style="color:white; text-decoration:none;">Configurações</a></div>
    <div class="menu-item">⚙️ <a href="lista_funcionarios.php" style="color:white; text-decoration:none;">Avaliação de Desempenho</a></div>
    <div class="menu-item">🔑 <a href="alterar_senha.php" style="color:white; text-decoration:none;">Alterar Senha</a></div>
    <div class="menu-item">🚪 <a href="login.php" style="color:white; text-decoration:none;">Sair</a></div>
</div>

<div class="main-content">
    <div class="statistics-container">
        <div class="stat-card" style="background-color: #ffc107; color: #000;">
            <h3>Total de Alunos</h3>
            <p><?= htmlspecialchars($qtdAlunos) ?></p>
        </div>
        <div class="stat-card" style="background-color: #28a745; color: white;">
            <h3>Total de Usuários</h3>
            <p><?= htmlspecialchars($qtdUsuarios) ?></p>
        </div>
        <div class="stat-card" style="background-color: #ffc107; color: #000;">
            <h3>Total de Turmas</h3>
            <p><?= htmlspecialchars($qtdTurmas) ?></p>
        </div>
    </div>

    <?php if (!empty($mensagem_upload)): ?>
        <div class="mensagem <?= strpos($mensagem_upload, '✅') !== false ? 'sucesso' : 'erro' ?>">
            <?= $mensagem_upload ?>
        </div>
    <?php endif; ?>

    <div class="graficos-container">
        <div class="grafico-bloco">
            <h3 style="color:#003366; margin-bottom: 10px;">Distribuição de Alunos por Classe</h3>
            <canvas id="graficoClasse"></canvas>
        </div>
        <div class="grafico-bloco">
            <h3 style="color:#003366; margin-bottom: 10px;">Evolução de Notas por Classe</h3>
            <canvas id="graficoNotas"></canvas>
        </div>
    </div>
</div>

<script>
// Função para mostrar/esconder a barra lateral em telas pequenas
function toggleMenu() {
    const sidebar = document.getElementById('sidebar');
    sidebar.classList.toggle('active');
    const mainContent = document.querySelector('.main-content');
    const topbar = document.querySelector('.topbar');
    if (sidebar.classList.contains('active')) {
        mainContent.style.marginLeft = '250px';
        topbar.style.left = '250px';
    } else {
        mainContent.style.marginLeft = '0';
        topbar.style.left = '0';
    }
}

// Gráfico de Pizza: Distribuição de Alunos por Classe (Dinâmico)
const ctx = document.getElementById('graficoClasse');
new Chart(ctx, {
    type: 'pie',
    data: {
        // Os dados dos rótulos e valores são injetados diretamente do PHP.
        labels: <?= json_encode($labels_classe) ?>,
        datasets: [{
            label: 'Distribuição',
            data: <?= json_encode($data_classe) ?>,
            backgroundColor: ['#007bff', '#28a745', '#ffc107', '#dc3545', '#6f42c1', '#17a2b8', '#fd7e14', '#e83e8c']
        }]
    },
    options: {
        responsive: true,
        plugins: {
            legend: { position: 'bottom' }
        }
    }
});

// Gráfico de Barras: Evolução de Notas por Classe (Dinâmico)
const notasCtx = document.getElementById('graficoNotas');
new Chart(notasCtx, {
    type: 'bar',
    data: {
        labels: <?= json_encode($labels_notas) ?>,
        datasets: <?= json_encode($datasets_notas) ?>
    },
    options: {
        responsive: true,
        plugins: {
            legend: { position: 'top' }
        },
        scales: {
            y: {
                beginAtZero: true,
                title: { display: true, text: 'Média das Notas' }
            }
        }
    }
});
</script>

</body>
</html>
