<?php
session_start();
include 'conexao.php';

// Verifica se o usuário é da Secretaria Pedagógica
if (!isset($_SESSION['usuario']) || strtoupper($_SESSION['usuario']['perfil']) !== 'SECRETARIA PEDAGÓGICA') {
    header("Location: login.php");
    exit;
}

$usuario = $_SESSION['usuario'];
$nome = htmlspecialchars($usuario['nome'] ?? 'Funcionário(a)');
$foto = htmlspecialchars($usuario['foto'] ?? 'sem-foto.png');
$mensagem_upload = '';

// --- LÓGICA DE UPLOAD DE FOTO ---
if (isset($_POST['upload_foto'])) {
    if (isset($_FILES['nova_foto']) && $_FILES['nova_foto']['error'] === 0) {
        $diretorio_destino = 'fotos/';
        $nome_original = basename($_FILES['nova_foto']['name']);
        $extensao = pathinfo($nome_original, PATHINFO_EXTENSION);
        $novo_nome_foto = uniqid() . '.' . $extensao;
        $caminho_completo = $diretorio_destino . $novo_nome_foto;

        // Tenta mover o arquivo para o diretório de destino
        if (move_uploaded_file($_FILES['nova_foto']['tmp_name'], $caminho_completo)) {
            // Atualiza o nome da foto no banco de dados
            $id_usuario = $usuario['id'] ?? null; // Assumindo que a sessão tem o ID do usuário
            if ($id_usuario) {
                // Prepara a consulta para evitar injeção de SQL
                $stmt = $conn->prepare("UPDATE usuarios SET foto = ? WHERE id = ?");
                $stmt->bind_param("si", $novo_nome_foto, $id_usuario);
                
                if ($stmt->execute()) {
                    // Atualiza a foto na sessão
                    $_SESSION['usuario']['foto'] = $novo_nome_foto;
                    $foto = htmlspecialchars($_SESSION['usuario']['foto']);
                    $mensagem_upload = 'Foto atualizada com sucesso!';
                } else {
                    $mensagem_upload = 'Erro ao atualizar o banco de dados.';
                }
                $stmt->close();
            } else {
                $mensagem_upload = 'ID do usuário não encontrado na sessão.';
            }
        } else {
            $mensagem_upload = 'Erro ao mover a foto para o diretório de destino.';
        }
    } else {
        $mensagem_upload = 'Erro no upload do arquivo.';
    }
}

// --- GRÁFICO DE PIZZA: Alunos por Gênero ---
// Esta consulta conta o número de alunos por gênero (masculino, feminino, outro)
$sql = "SELECT genero, COUNT(*) as total FROM alunos GROUP BY genero";
$result = $conn->query($sql);

$generos = [];
if ($result && $result->num_rows > 0) {
    while ($row = $result->fetch_assoc()) {
        $generos[$row['genero']] = (int)$row['total'];
    }
}
$labelsAlunos = array_keys($generos);
$dataAlunos = array_values($generos);
$totalAlunos = array_sum($dataAlunos);

// --- GRÁFICO DE COLUNAS: Alunos por Turma ---
// Esta consulta conta o número de alunos matriculados em cada turma
$sql = "SELECT t.nome AS turma, COUNT(m.aluno_id) as total
        FROM turmas t
        LEFT JOIN matriculas m ON t.id = m.turma_id
        GROUP BY t.nome";
$result = $conn->query($sql);

$turmas = [];
if ($result && $result->num_rows > 0) {
    while ($row = $result->fetch_assoc()) {
        $turmas[$row['turma']] = (int)$row['total'];
    }
}
$labelsTurmas = array_keys($turmas);
$dataTurmas = array_values($turmas);

// --- GRÁFICO DE BARRAS: Média por Disciplina ---
// Esta consulta calcula a média das notas para cada disciplina
$sql = "SELECT d.nome AS disciplina, AVG(n.mt) as media
        FROM notas n
        INNER JOIN disciplinas d ON n.disciplina_id = d.id
        GROUP BY d.nome";
$result = $conn->query($sql);

$disciplinas = [];
$medias = [];
if ($result && $result->num_rows > 0) {
    while ($row = $result->fetch_assoc()) {
        $disciplinas[] = $row['disciplina'];
        $medias[] = round($row['media'], 1);
    }
}
?>

<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <title>Painel Secretaria Pedagógica</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="adminlte/plugins/fontawesome-free/css/all.min.css">
    <link rel="stylesheet" href="adminlte/dist/css/adminlte.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

    <style>
        body { margin: 0; font-family: 'Segoe UI', sans-serif; background-color: #f4f6f9; }
        .sidebar { width: 260px; background: #001f3f; color: white; position: fixed; top: 0; bottom: 0; padding-top: 60px; }
        .sidebar a { display: block; color: white; padding: 15px 20px; text-decoration: none; }
        .sidebar a:hover { background: #0056b3; }
        .topbar { position: fixed; left: 260px; right: 0; top: 0; height: 60px; background: #003366; color: white; display: flex; align-items: center; justify-content: space-between; padding: 0 20px; z-index: 1000; }
        .topbar .user-info { display: flex; align-items: center; gap: 10px; }
        .topbar img { width: 40px; height: 40px; border-radius: 50%; object-fit: cover; border: 2px solid white; }
        .main-content { margin-left: 260px; margin-top: 60px; padding: 30px; }
        .grafico-container {
            background: white;
            border-radius: 10px;
            padding: 20px;
            margin-top: 20px;
            display: none;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }

        /* Estilo da mensagem de sucesso/erro */
        .mensagem {
            margin-top: 10px;
            font-weight: bold;
            padding: 10px;
            border-radius: 5px;
            text-align: center;
        }
        .mensagem.sucesso {
            color: #28a745;
            background-color: #d4edda;
            border: 1px solid #c3e6cb;
        }
        .mensagem.erro {
            color: #dc3545;
            background-color: #f8d7da;
            border: 1px solid #f5c6cb;
        }
    </style>
</head>
<body>

<!-- Sidebar -->
<div class="sidebar">
    <a href="mapa_aproveitamento.php"><i class="fas fa-table"></i> Mapas de Aproveitamento</a>
    <a href="mini_pauta.php"><i class="fas fa-table"></i> Gestão de Mini Pautas</a>
    <a href="planificacao_modelo.php"><i class="fas fa-table"></i> Modelo de Planificação</a>
    <a href="ver_turmas.php"><i class="fas fa-table"></i> Cadastrar Dados do Aluno</a>
    <a href="ver_notas_finais.php"><button class="btn btn-primary">Ver Notas Finais</button></a>
    <a href="painel_certificados.php"><i class="fas fa-certificate"></i> Certificados</a>
    <a href="pauta_final_aluno.php"><i class="fas fa-certificate"></i> Termo de Frequência e Exame</a>
    <a href="alterar_senha.php"><i class="fas fa-lock"></i> Alterar Senha</a>
    <a href="enviar_documento.php"><i class="fas fa-upload"></i> Enviar Documento</a>
    <a href="#" onclick="mostrarGrafico('coluna')"><i class="fas fa-chart-bar"></i> Gráfico por Turma</a>
    <a href="#" onclick="mostrarGrafico('pizza')"><i class="fas fa-chart-pie"></i> Gráfico por Alunos</a>
    <a href="#" onclick="mostrarGrafico('barra')"><i class="fas fa-chart-bar"></i> Gráfico por Disciplina</a>
    <a href="login.php"><i class="fas fa-sign-out-alt"></i> Sair</a>
</div>

<!-- Topbar -->
<div class="topbar">
    <div><strong>PAINEL DA SECRETARIA PEDAGÓGICA</strong> | Bem-vindo, <?= htmlspecialchars($nome) ?></div>
    <div class="user-info">
        <!-- Adiciona um parâmetro de tempo para evitar o cache do navegador -->
        <img src="fotos/<?= htmlspecialchars($foto) ?>?t=<?= time() ?>" alt="Foto do Usuário">
    </div>
</div>

<!-- Conteúdo -->
<div class="main-content">
    <h2>Bem-vindo(a), <?= htmlspecialchars($nome) ?>!</h2>
    <p>Selecione uma funcionalidade na barra lateral.</p>

    <!-- Exibe a mensagem de upload -->
    <?php if (!empty($mensagem_upload)): ?>
        <div class="mensagem <?= strpos($mensagem_upload, 'sucesso') !== false ? 'sucesso' : 'erro' ?>">
            <?= $mensagem_upload ?>
        </div>
    <?php endif; ?>

    <!-- Seção de upload de foto -->
    <div class="card">
        <div class="card-body">
            <h3>Atualizar Foto de Perfil</h3>
            <form action="" method="post" enctype="multipart/form-data">
                <div class="form-group">
                    <input type="file" name="nova_foto" class="form-control-file" required>
                </div>
                <button type="submit" name="upload_foto" class="btn btn-primary">Upload Foto</button>
            </form>
        </div>
    </div>

    <!-- Gráfico de Alunos (Pizza) -->
    <div id="grafico-pizza" class="grafico-container">
        <h4>Distribuição de Alunos por Gênero</h4>
        <canvas id="pizzaChart" height="200"></canvas>
    </div>

    <!-- Gráfico de Turmas (Coluna) -->
    <div id="grafico-coluna" class="grafico-container">
        <h4>Número de Alunos por Turma</h4>
        <canvas id="colunaChart" height="100"></canvas>
    </div>

    <!-- Gráfico de Disciplinas (Barra) -->
    <div id="grafico-barra" class="grafico-container">
        <h4>Média por Disciplina</h4>
        <canvas id="barraChart" height="100"></canvas>
    </div>
</div>

<!-- Script -->
<script>
let colunaChart = null;
let pizzaChart = null;
let barraChart = null;

function mostrarGrafico(tipo) {
    document.getElementById('grafico-coluna').style.display = 'none';
    document.getElementById('grafico-pizza').style.display = 'none';
    document.getElementById('grafico-barra').style.display = 'none';

    if (tipo === 'coluna') {
        document.getElementById('grafico-coluna').style.display = 'block';
        renderizarColuna();
    } else if (tipo === 'pizza') {
        document.getElementById('grafico-pizza').style.display = 'block';
        renderizarPizza();
    } else if (tipo === 'barra') {
        document.getElementById('grafico-barra').style.display = 'block';
        renderizarBarra();
    }
}

function renderizarColuna() {
    if (colunaChart) colunaChart.destroy();
    const ctx = document.getElementById('colunaChart').getContext('2d');
    const labels = <?= json_encode($labelsTurmas) ?>;
    const data = <?= json_encode($dataTurmas) ?>;

    colunaChart = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: labels,
            datasets: [{
                label: 'Número de Alunos',
                data: data,
                backgroundColor: '#007bff'
            }]
        },
        options: {
            responsive: true,
            plugins: {
                legend: { display: false },
                title: { display: true, text: 'Número de Alunos por Turma' }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    title: {
                        display: true,
                        text: 'Número de Alunos'
                    }
                }
            }
        }
    });
}

function renderizarPizza() {
    if (pizzaChart) pizzaChart.destroy();
    const ctx = document.getElementById('pizzaChart').getContext('2d');
    const labels = <?= json_encode($labelsAlunos) ?>;
    const data = <?= json_encode($dataAlunos) ?>;

    pizzaChart = new Chart(ctx, {
        type: 'pie',
        data: {
            labels: labels,
            datasets: [{
                label: 'Total de Alunos',
                data: data,
                backgroundColor: [
                    '#007bff', // Azul para Masculino
                    '#dc3545', // Vermelho para Feminino
                    '#ffc107'  // Amarelo para Outro
                ]
            }]
        },
        options: {
            responsive: true,
            plugins: {
                legend: {
                    position: 'top',
                },
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            let label = context.label || '';
                            if (label) {
                                label += ': ';
                            }
                            const value = context.parsed;
                            const total = context.dataset.data.reduce((sum, current) => sum + current, 0);
                            const percentage = ((value / total) * 100).toFixed(1) + '%';
                            return label + value + ' (' + percentage + ')';
                        }
                    }
                }
            }
        }
    });
}

function renderizarBarra() {
    if (barraChart) barraChart.destroy();
    const ctx = document.getElementById('barraChart').getContext('2d');
    
    // Dados agora são buscados do PHP
    const labels = <?= json_encode($disciplinas) ?>;
    const data = <?= json_encode($medias) ?>;
    
    barraChart = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: labels,
            datasets: [{
                label: 'Média por Disciplina',
                data: data,
                backgroundColor: ['#28a745', '#ffc107', '#17a2b8', '#dc3545']
            }]
        },
        options: {
            responsive: true,
            indexAxis: 'y',
            plugins: {
                legend: { display: false },
                title: { display: true, text: 'Média por Disciplina' }
            },
            scales: {
                x: {
                    beginAtZero: true,
                    max: 20
                }
            }
        }
    });
}
</script>

</body>
</html>