<?php
session_start();
include 'conexao.php';

// Verifica a permissão do usuário
if (!isset($_SESSION['usuario']) || strtoupper($_SESSION['usuario']['perfil']) !== 'SUBDIRECÇÃO PEDAGÓGICA') {
    header("Location: login.php");
    exit;
}

// Lógica de salvamento de notas (POST) foi removida para tornar a pauta apenas visualizável.

// Busca anos letivos disponíveis
$anos_letivos_query = $conn->query("SELECT DISTINCT ano FROM turmas ORDER BY ano DESC");
$anos_letivos = $anos_letivos_query->fetch_all(MYSQLI_ASSOC);

$turmas = [];
$turma_id = isset($_GET['turma_id']) ? intval($_GET['turma_id']) : 0;
$ano_letivo_selecionado = isset($_GET['ano_letivo']) ? $_GET['ano_letivo'] : '';

// Se um ano letivo for selecionado, busca as turmas correspondentes
if ($ano_letivo_selecionado) {
    $stmt = $conn->prepare("SELECT id, nome, classe, curso, turno FROM turmas WHERE ano = ? ORDER BY nome");
    $stmt->bind_param("s", $ano_letivo_selecionado);
    $stmt->execute();
    $turmas = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

$dados_alunos = [];
$disciplinas = [];
$info_turma = [];

if ($turma_id > 0) {
    // 1. Busca informações da turma
    $stmt = $conn->prepare("SELECT * FROM turmas WHERE id = ?");
    $stmt->bind_param("i", $turma_id);
    $stmt->execute();
    $info_turma = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    // 2. Busca disciplinas da turma, ordenando por prioridade
    $stmt = $conn->prepare("
        SELECT d.id, d.nome, d.tipo
        FROM disciplinas d
        JOIN atribuicoes a ON d.id = a.disciplina_id
        WHERE a.turma_id = ?
        GROUP BY d.id
        ORDER BY
            CASE
                WHEN d.nome = 'LÍNGUA PORTUGUESA' THEN 0
                WHEN d.nome = 'INGLÊS' THEN 1
                WHEN d.nome = 'MATEMÁTICA' THEN 2
                WHEN d.tipo = 'nuclear' THEN 3
                ELSE 4
            END,
            d.nome
    ");
    $stmt->bind_param("i", $turma_id);
    $stmt->execute();
    $disciplinas = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // 3. Busca todos os alunos da turma
    $stmt_alunos = $conn->prepare("
        SELECT a.id, a.nome, m.observacoes
        FROM alunos a
        JOIN matriculas m ON a.id = m.aluno_id
        WHERE m.turma_id = ? AND m.ano_letivo = ?
        ORDER BY a.nome
    ");
    $stmt_alunos->bind_param("is", $turma_id, $ano_letivo_selecionado);
    $stmt_alunos->execute();
    $alunos = $stmt_alunos->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt_alunos->close();

    // 4. Busca todas as notas de uma vez (consolidada no trimestre 3)
    $stmt_notas = $conn->prepare("
        SELECT
            aluno_id, disciplina_id, mt1, mt2, mt3, mfd, neo, nee, mec, mf
        FROM notas
        WHERE turma_id = ? AND ano_lectivo = ? AND trimestre = 3
    ");
    $stmt_notas->bind_param("is", $turma_id, $ano_letivo_selecionado);
    $stmt_notas->execute();
    $result_notas = $stmt_notas->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt_notas->close();

    // 5. Organiza as notas para fácil acesso
    $notas_por_aluno_disciplina = [];
    foreach ($result_notas as $nota_row) {
        $aluno_id = $nota_row['aluno_id'];
        $disc_id = $nota_row['disciplina_id'];

        $notas_por_aluno_disciplina[$aluno_id][$disc_id] = [
            'mt1' => (float)($nota_row['mt1'] ?? 0),
            'mt2' => (float)($nota_row['mt2'] ?? 0),
            'mt3' => (float)($nota_row['mt3'] ?? 0),
            'mfd' => (float)($nota_row['mfd'] ?? 0),
            'neo' => (float)($nota_row['neo'] ?? 0),
            'nee' => (float)($nota_row['nee'] ?? 0),
            'mec' => (float)($nota_row['mec'] ?? 0),
            'mf'  => (float)($nota_row['mf'] ?? 0)
        ];
    }

    // 6. Lógica para calcular e preparar dados para exibição
    $n_ordem = 1;
    foreach ($alunos as $aluno) {
        $aluno_id = $aluno['id'];
        $linha = [
            'id' => $aluno_id,
            'n_ordem' => $n_ordem++,
            'nome' => $aluno['nome'],
            'notas_finais' => [],
            'neg_mfd_count' => 0,
            'neg_mfd_nuclear_count' => 0,
            'neg_mfd_nao_nuclear_count' => 0,
            'has_nuclear_negative_mfd' => false,
            'media_geral' => 0,
            'observacao' => $aluno['observacao_final'] ?? ''
        ];

        $total_mfd_para_media_geral = 0;
        $qtd_mfd_para_media_geral = 0;

        foreach ($disciplinas as $d) {
            $disc_id = $d['id'];
            $notas_raw = $notas_por_aluno_disciplina[$aluno_id][$disc_id] ?? [];

            $mt1_val_raw = $notas_raw['mt1'] ?? 0;
            $mt2_val_raw = $notas_raw['mt2'] ?? 0;
            $mt3_val_raw = $notas_raw['mt3'] ?? 0;
            $neo_val_raw = $notas_raw['neo'] ?? 0;
            $nee_val_raw = $notas_raw['nee'] ?? 0;

            $mfd_calc = ($mt1_val_raw + $mt2_val_raw + $mt3_val_raw) / 3;
            $mfd_calc = round($mfd_calc, 2);
            $mfd_display = ($mfd_calc == 0 && $mt1_val_raw == 0 && $mt2_val_raw == 0 && $mt3_val_raw == 0) ? '' : number_format($mfd_calc, 2, '.', '');

            $mec_calc = null;
            $mf_calc = null;

            if ($info_turma['classe'] == 12) {
                if (mb_strtoupper($d['nome']) === 'LÍNGUA PORTUGUESA') {
                    if ($nee_val_raw > 0 || $neo_val_raw > 0) {
                        $mec_calc = ($nee_val_raw + $neo_val_raw) / 2;
                        $mec_calc = round($mec_calc, 2);
                    } else {
                        $mec_calc = null;
                    }
                    if ($mfd_calc > 0 && $mec_calc !== null && $mec_calc > 0) {
                        $mf_calc = ($mfd_calc * 0.6) + ($mec_calc * 0.4);
                    } else {
                        $mf_calc = $mfd_calc;
                    }
                } else {
                    if ($mfd_calc > 0 && $nee_val_raw > 0) {
                        $mf_calc = ($mfd_calc * 0.6) + ($nee_val_raw * 0.4);
                    } else {
                        $mf_calc = $mfd_calc;
                    }
                }
            } else {
                $mf_calc = $mfd_calc;
            }

            $mf_calc = is_numeric($mf_calc) ? round($mf_calc) : null;
            $mf_display = ($mf_calc !== null && $mf_calc > 0 ? $mf_calc : '');
            if ($mf_calc == 0 && $mfd_calc == 0 && $mt1_val_raw == 0 && $mt2_val_raw == 0 && $mt3_val_raw == 0) $mf_display = '';

            if (is_numeric($mfd_calc) && $mfd_calc > 0 && $mfd_calc < 10) {
                $linha['neg_mfd_count']++;
                if ($d['tipo'] === 'nuclear') {
                    $linha['neg_mfd_nuclear_count']++;
                    $linha['has_nuclear_negative_mfd'] = true;
                } else {
                    $linha['neg_mfd_nao_nuclear_count']++;
                }
            }

            $linha['notas_finais'][$d['id']] = [
                'mt1' => ($mt1_val_raw > 0 ? number_format($mt1_val_raw, 2, '.', '') : ''),
                'mt2' => ($mt2_val_raw > 0 ? number_format($mt2_val_raw, 2, '.', '') : ''),
                'mt3' => ($mt3_val_raw > 0 ? number_format($mt3_val_raw, 2, '.', '') : ''),
                'mfd' => $mfd_display,
                'neo' => ($neo_val_raw > 0 ? number_format($neo_val_raw, 2, '.', '') : ''),
                'nee' => ($nee_val_raw > 0 ? number_format($nee_val_raw, 2, '.', '') : ''),
                'mec' => ($mec_calc !== null && $mec_calc > 0 ? number_format($mec_calc, 2, '.', '') : ''),
                'mf'  => $mf_display
            ];

            if (is_numeric($mfd_calc) && $mfd_calc > 0) {
                $total_mfd_para_media_geral += $mfd_calc;
                $qtd_mfd_para_media_geral++;
            }
        }

        $linha['total_neg'] = $linha['neg_mfd_count'];
        $linha['media_geral'] = $qtd_mfd_para_media_geral > 0 ? number_format($total_mfd_para_media_geral / $qtd_mfd_para_media_geral, 1, '.', '') : '-';

        $observacao_auto = 'NÃO DEFINIDO';
        if ($linha['has_nuclear_negative_mfd']) {
            $observacao_auto = 'NÃO TRANSITA';
        } elseif ($linha['neg_mfd_count'] > 3) {
            $observacao_auto = 'NÃO TRANSITA';
        } elseif ($linha['neg_mfd_count'] >= 1 && $linha['neg_mfd_count'] <= 3) {
            $observacao_auto = 'CONSELHO';
        } elseif ($linha['neg_mfd_count'] == 0) {
            $observacao_auto = 'TRANSITA';
        }
        
        if (empty($aluno['observacao_final'])) {
            $linha['observacao'] = $observacao_auto;
        }

        $dados_alunos[] = $linha;
    }
}
?>

<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <title>Pauta Final</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;600&display=swap" rel="stylesheet">
    <style>
        /* Estilos CSS Gerais */
        body { 
            font-family: 'Poppins', sans-serif; 
            margin: 0; 
            padding: 0; 
            background-color: #f4f7f9; 
            color: #333; 
            font-size: 12px;
        }
        .container { 
            width: 98%;
            margin: 20px auto; 
            background-color: #fff; 
            padding: 20px; 
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.1); 
            border-radius: 8px; 
        }
        .form-container { text-align: center; margin-bottom: 20px; }
        .form-container select { padding: 8px; border-radius: 5px; border: 1px solid #ccc; margin: 0 5px; font-size: 14px; }
        .pauta-document { border: 1px solid #ccc; padding: 20px; border-radius: 8px; margin-top: 20px; position: relative; }
        .director-signature { 
            position: absolute; 
            top: 20px; 
            left: 20px; 
            text-align: center; 
            width: 150px; 
            border: 1px solid #000; 
            display: flex; 
            flex-direction: column; 
            align-items: center; 
            justify-content: center; 
            font-size: 10px; 
            line-height: 1.2; 
            padding: 5px; 
        }
        .header-institutional { text-align: center; margin-bottom: 20px; padding-top: 120px; position: relative; }
        .insignia-box { 
            position: absolute; 
            top: 0; 
            left: 50%; 
            transform: translateX(-50%); 
            width: 100px; 
            height: 100px; 
            background-image: url('insignia/insignia.png');
            background-size: contain; 
            background-repeat: no-repeat; 
            background-position: center; 
        }
        .header-institutional h3 { margin: 0; font-size: 16px; }
        .header-institutional p { margin: 0; font-size: 12px; }
        .info-turma { text-align: center; font-weight: bold; margin-bottom: 15px; font-size: 14px; }
        .print-button { text-align: right; margin-bottom: 10px; }
        .print-button button { padding: 8px 15px; background-color: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer; font-size: 12px; }
        .pauta-table-container { overflow-x: auto; }
        
        /* Ajuste da tabela para não estreitar colunas */
        .pauta-table { 
            width: 100%; 
            border-collapse: collapse; 
            font-size: 12px;
            table-layout: auto; /* Altera para layout automático */
        }
        .pauta-table thead th, .pauta-table tbody td { 
            border: 1px solid #000; 
            padding: 12px 5px; /* Reduz um pouco o padding para caber mais info */
            text-align: center; 
            vertical-align: middle; 
            white-space: nowrap; /* Evita que o texto quebre a linha */
        }
        .main-separator { border-right: 2px solid #000 !important; }
        .bottom-separator th { border-bottom: 2px solid #000 !important; }
        .pauta-table thead th { background-color: #e9ecef; font-weight: bold; color: #212529; }
        .pauta-table tbody tr:nth-child(even) { background-color: #f8f9fa; }

        /* Ajustes de largura das colunas */
        .pauta-table .aluno-nome { 
            text-align: left; 
            padding-left: 10px; 
            background-color: #f0f0f0; 
            width: 250px; /* Largura fixa para nomes longos */
            min-width: 250px; /* Largura mínima para nomes */
        }
        .pauta-table .aluno-nome a { text-decoration: none; color: #333; font-weight: bold; }
        .pauta-table .aluno-nome a:hover { color: #0056b3; }
        
        .pauta-table .nota-cell {
            min-width: 60px; /* Largura mínima para células de nota */
        }

        /* Estilos de cor para as notas */
        .pauta-table .nota-baixa { 
            color: #dc3545; 
            font-weight: bold; 
            background-color: #ffcccc; 
        }
        .pauta-table .nota-suficiente { 
            color: #28a745; 
            font-weight: bold; 
            background-color: #e6ffcc; 
        }

        /* Estilos da Observação */
        .observacao-aprovado { background-color: #d4edda; color: #155724; font-weight: bold; }
        .observacao-conselho { background-color: #fff3cd; color: #856404; font-weight: bold; }
        .observacao-reprovado { background-color: #f8d7da; color: #721c24; font-weight: bold; }

        .rotate { height: 120px; white-space: nowrap; }
        .rotate div { 
            transform: rotate(-90deg); 
            width: 20px;
        }
        .footer-signatures { display: flex; justify-content: space-between; margin-top: 50px; }
        .signatures-left, .signatures-right { text-align: center; width: 45%; }
        .signature-line { width: 100%; height: 1px; background-color: #000; margin: 5px auto; }
        .signature-name { font-weight: bold; margin-top: 5px; }
        .footer-date-place { text-align: center; margin-top: 20px; font-size: 12px; }

        /* Media Query para impressão */
        @media print {
            body { 
                background-color: #fff; 
                -webkit-print-color-adjust: exact !important;
                color-adjust: exact !important; 
            }
            .container { box-shadow: none; border: none; margin: 0; }
            .no-print { display: none; }
            .pauta-document { border: none; padding: 0; }
            .pauta-table { 
                width: 100%; 
                border-collapse: collapse; 
                border: 1px solid #000 !important;
            }
            .pauta-table thead th, 
            .pauta-table tbody td { 
                border: 1px solid #000 !important;
                padding: 12px 5px;
                background-color: transparent !important; 
                white-space: nowrap;
            }
            .pauta-table thead th { background-color: #e9ecef !important; }

            .pauta-table .nota-baixa {
                -webkit-print-color-adjust: exact !important;
                color-adjust: exact !important;
                color: #dc3545 !important;
                font-weight: bold;
                background-color: #ffcccc !important;
            }
            .pauta-table .nota-suficiente {
                -webkit-print-color-adjust: exact !important;
                color-adjust: exact !important;
                color: #28a745 !important;
                font-weight: bold;
                background-color: #e6ffcc !important;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="form-container no-print">
            <form action="" method="get">
                <label for="ano_letivo">Selecione o Ano Lectivo:</label>
                <select name="ano_letivo" id="ano_letivo" onchange="this.form.submit()">
                    <option value="">- Selecionar Ano -</option>
                    <?php foreach ($anos_letivos as $a): ?>
                        <option value="<?= htmlspecialchars($a['ano']) ?>" <?= ($a['ano'] == $ano_letivo_selecionado) ? 'selected' : '' ?>>
                            <?= htmlspecialchars($a['ano']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <?php if ($ano_letivo_selecionado): ?>
                    <label for="turma_id">Selecione a Turma:</label>
                    <select name="turma_id" id="turma_id" onchange="this.form.submit()">
                        <option value="0">- Selecionar Turma -</option>
                        <?php foreach ($turmas as $t): ?>
                            <option value="<?= $t['id'] ?>" <?= ($t['id'] == $turma_id) ? 'selected' : '' ?>>
                                <?= $t['classe'] ?>ª - <?= $t['nome'] ?> - <?= $t['curso'] ?> - <?= $t['turno'] ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                <?php endif; ?>
            </form>
        </div>
        <?php if ($turma_id > 0 && !empty($info_turma)): ?>
            <div class="pauta-document">
                <div class="director-signature">
                    <p style="margin: 0;">O DIRECTOR DO COMPLEXO ESCOLAR</p>
                    <p style="margin: 5px 0; font-size: 8px;">..............................</p>
                    <p style="margin: 0; font-weight: bold;">Eduardo Sanjambela Tchandja</p>
                    <p style="margin: 5px 0; font-size: 8px;">_____ / _____ / _____</p>
                </div>
                <div class="header-institutional">
                    <div class="insignia-box"></div>
                    <h3>REPÚBLICA DE ANGOLA</h3>
                    <p>Governo Provincial de Benguela</p>
                    <p>Gabinete Provincial da Educação</p>
                    <h3>COMPLEXO ESCOLAR BG0026 PADRE EMIL FRICK - CATUMBELA</h3>
                    <h3>PAUTA DE AVALIAÇÃO ANUAL</h3>
                </div>
                <div class="info-turma">
                    ANO LECTIVO: <?= htmlspecialchars($info_turma['ano']) ?> |
                    <?= htmlspecialchars($info_turma['classe']) ?> CLASSE |
                    CURSO: <?= htmlspecialchars($info_turma['curso']) ?> |
                    TURMA: <?= htmlspecialchars($info_turma['nome']) ?> |
                    PERÍODO: <?= ucfirst(strtolower($info_turma['turno'])) ?>
                </div>
                <div class="print-button no-print">
                    <button onclick="window.print()">🖨️ Imprimir</button>
                </div>
                <div class="pauta-table-container">
                    <table class="pauta-table">
                        <thead>
                            <tr>
                                <th rowspan="3" class="main-separator">Nº</th>
                                <th rowspan="3" class="aluno-nome main-separator">Nome do Aluno</th>
                                <?php foreach ($disciplinas as $d):
                                    $is_12_LP = ($info_turma['classe'] == 12 && mb_strtoupper($d['nome']) === 'LÍNGUA PORTUGUESA');
                                    $is_12_other = ($info_turma['classe'] == 12 && mb_strtoupper($d['nome']) !== 'LÍNGUA PORTUGUESA');
                                    $colspan_val = 0;
                                    if ($is_12_LP) {
                                        $colspan_val = 3;
                                    } elseif ($is_12_other) {
                                        $colspan_val = 3;
                                    } else {
                                        $colspan_val = 4;
                                    }
                                ?>
                                    <th colspan="<?= $colspan_val ?>" class="main-separator"><?= htmlspecialchars($d['nome']) ?></th>
                                <?php endforeach; ?>
                                <th rowspan="3" class="rotate main-separator"><div>Neg. MFD (Total)</div></th>
                                <th rowspan="3" class="rotate main-separator"><div>Neg. MFD (Nuclear)</div></th>
                                <th rowspan="3" class="rotate main-separator"><div>Neg. MFD (Não Nuclear)</div></th>
                                <th rowspan="3" class="rotate main-separator"><div>Média Geral (MFD)</div></th>
                                <th rowspan="3">Observação</th>
                            </tr>
                            <tr>
                                <?php foreach ($disciplinas as $d):
                                    $is_12_LP = ($info_turma['classe'] == 12 && mb_strtoupper($d['nome']) === 'LÍNGUA PORTUGUESA');
                                    $is_12_other = ($info_turma['classe'] == 12 && mb_strtoupper($d['nome']) !== 'LÍNGUA PORTUGUESA');
                                    $colspan_val = 0;
                                    if ($is_12_LP || $is_12_other) {
                                        $colspan_val = 3;
                                    } else {
                                        $colspan_val = 4;
                                    }
                                ?>
                                    <th colspan="<?= $colspan_val ?>" class="main-separator"></th>
                                <?php endforeach; ?>
                            </tr>
                            <tr class="bottom-separator">
                                <?php foreach ($disciplinas as $d):
                                    $is_12_LP = ($info_turma['classe'] == 12 && mb_strtoupper($d['nome']) === 'LÍNGUA PORTUGUESA');
                                    $is_12_other = ($info_turma['classe'] == 12 && mb_strtoupper($d['nome']) !== 'LÍNGUA PORTUGUESA');
                                ?>
                                    <?php if ($is_12_other): ?>
                                        <th class="nota-cell">MFD</th><th class="nota-cell">NEE</th><th class="main-separator nota-cell">MF</th>
                                    <?php elseif ($is_12_LP): ?>
                                        <th class="nota-cell">MFD</th><th class="nota-cell">MEC</th><th class="main-separator nota-cell">MF</th>
                                    <?php else: ?>
                                        <th class="nota-cell">MT1</th><th class="nota-cell">MT2</th><th class="nota-cell">MT3</th><th class="main-separator nota-cell">MFD</th>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($dados_alunos as $aluno): ?>
                                <tr>
                                    <td class="main-separator"><?= $aluno['n_ordem'] ?></td>
                                    <td class="aluno-nome main-separator">
                                        <a href="ver_termo.php?aluno_id=<?= $aluno['id'] ?>&turma_id=<?= $turma_id ?>&ano_letivo=<?= urlencode($info_turma['ano'] ?? '') ?>">
                                            <?= htmlspecialchars($aluno['nome']) ?>
                                        </a>
                                    </td>
                                    <?php foreach ($disciplinas as $d):
                                        $notas_formatadas = $aluno['notas_finais'][$d['id']] ?? [];
                                        $is_12_LP = ($info_turma['classe'] == 12 && mb_strtoupper($d['nome']) === 'LÍNGUA PORTUGUESA');
                                        $is_12_other = ($info_turma['classe'] == 12 && mb_strtoupper($d['nome']) !== 'LÍNGUA PORTUGUESA');
                                        $mfd_val = (float)($notas_formatadas['mfd'] ?? 0);
                                        $mf_val = (float)($notas_formatadas['mf'] ?? 0);
                                        $nee_val = (float)($notas_formatadas['nee'] ?? 0);
                                        $mec_val = (float)($notas_formatadas['mec'] ?? 0);
                                        $mt1_val = (float)($notas_formatadas['mt1'] ?? 0);
                                        $mt2_val = (float)($notas_formatadas['mt2'] ?? 0);
                                        $mt3_val = (float)($notas_formatadas['mt3'] ?? 0);
                                    ?>
                                        <?php if ($is_12_other): ?>
                                            <td class="<?= ($mfd_val < 10 && $mfd_val > 0) ? 'nota-baixa' : 'nota-suficiente' ?>"><?= $notas_formatadas['mfd'] ?? '' ?></td>
                                            <td class="<?= ($nee_val < 10 && $nee_val > 0) ? 'nota-baixa' : 'nota-suficiente' ?>"><?= $notas_formatadas['nee'] ?? '' ?></td>
                                            <td class="main-separator <?= ($mf_val < 10 && $mf_val > 0) ? 'nota-baixa' : 'nota-suficiente' ?>"><?= $notas_formatadas['mf'] ?? '' ?></td>
                                        <?php elseif ($is_12_LP): ?>
                                            <td class="nota-cell <?= ($mfd_val < 10 && $mfd_val > 0) ? 'nota-baixa' : 'nota-suficiente' ?>"><?= $notas_formatadas['mfd'] ?? '' ?></td>
                                            <td class="nota-cell <?= ($mec_val < 10 && $mec_val > 0) ? 'nota-baixa' : 'nota-suficiente' ?>"><?= $notas_formatadas['mec'] ?? '' ?></td>
                                            <td class="main-separator nota-cell <?= ($mf_val < 10 && $mf_val > 0) ? 'nota-baixa' : 'nota-suficiente' ?>"><?= $notas_formatadas['mf'] ?? '' ?></td>
                                        <?php else: ?>
                                            <td class="nota-cell <?= ($mt1_val < 10 && $mt1_val > 0) ? 'nota-baixa' : 'nota-suficiente' ?>"><?= $notas_formatadas['mt1'] ?? '' ?></td>
                                            <td class="nota-cell <?= ($mt2_val < 10 && $mt2_val > 0) ? 'nota-baixa' : 'nota-suficiente' ?>"><?= $notas_formatadas['mt2'] ?? '' ?></td>
                                            <td class="nota-cell <?= ($mt3_val < 10 && $mt3_val > 0) ? 'nota-baixa' : 'nota-suficiente' ?>"><?= $notas_formatadas['mt3'] ?? '' ?></td>
                                            <td class="main-separator nota-cell <?= ($mfd_val < 10 && $mfd_val > 0) ? 'nota-baixa' : 'nota-suficiente' ?>"><?= $notas_formatadas['mfd'] ?? '' ?></td>
                                        <?php endif; ?>
                                    <?php endforeach; ?>
                                    <td class="main-separator"><?= $aluno['neg_mfd_count'] ?></td>
                                    <td class="main-separator"><?= $aluno['neg_mfd_nuclear_count'] ?></td>
                                    <td class="main-separator"><?= $aluno['neg_mfd_nao_nuclear_count'] ?></td>
                                    <td class="main-separator"><?= $aluno['media_geral'] ?></td>
                                    <td class="
                                        <?php 
                                            switch($aluno['observacao']) {
                                                case 'TRANSITA': echo 'observacao-aprovado'; break;
                                                case 'CONSELHO': echo 'observacao-conselho'; break;
                                                case 'NÃO TRANSITA': echo 'observacao-reprovado'; break;
                                                default: break;
                                            }
                                        ?>
                                    ">
                                        <?= htmlspecialchars($aluno['observacao']) ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <div class="footer-signatures no-print">
                    <div class="signatures-left">
                        <div class="signature-line"></div>
                        <div class="signature-name">O SUBDIRECTOR PEDAGÓGICO</div>
                        <div class="signature-line"></div>
                        <p>Delfim Alberto Tchandja</p>
                    </div>
                    <div class="signatures-right">
                        <div class="signature-line"></div>
                        <div class="signature-name">O SECRETÁRIO</div>
                        <div class="signature-line"></div>
                        <p>Beto João</p>
                    </div>
                </div>
                <div class="footer-date-place no-print">
                    <p>Catumbela, aos _____ de _________________ de ______</p>
                </div>
            </div>
        <?php else: ?>
            <p style="text-align: center;">Por favor, selecione um ano letivo e uma turma para visualizar a pauta.</p>
        <?php endif; ?>
    </div>
</body>
</html>