<?php
// =========================================================================
// preencher_avaliacao.php
// Objetivo: Interface para o Gestor preencher as pontuações dos indicadores
//           para um funcionário específico (por trimestre).
// =========================================================================

// --- CONFIGURAÇÃO DO BANCO DE DADOS ---
define('DB_HOST', 'localhost');
define('DB_NAME', 'sismag'); // Nome do BD corrigido
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_PORT', '3307'); 

$pdo = null;
$error_message = null;
$success_message = null;

// --- DADOS RECEBIDOS ---
$id_selecionado = isset($_GET['id']) ? (int)$_GET['id'] : null;
$ano_lectivo = $_GET['ano'] ?? date('Y');
$trimestre_focado = isset($_GET['trimestre']) ? (int)$_GET['trimestre'] : 1; 

// Mapa de Indicadores (Must be in sync with documento_oficial.php)
$indicadores_map = [
    1 => 'Qualidade do Processo de Ensino e Aprendizagem',
    2 => 'Progresso do Aluno ou Desenvolvimento do Aluno',
    3 => 'Aperfeiçoamento Profissional e Inovação Pedagógica',
    4 => 'Responsabilidade',
    5 => 'Relações Humanas',
];

if (!$id_selecionado || $trimestre_focado < 1 || $trimestre_focado > 3) {
    die("Erro: ID do funcionário ou Trimestre inválido. Use a lista de funcionários.");
}

try {
    // 1. Conexão com o Banco de Dados
    $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";port=" . DB_PORT . ";charset=utf8";
    $pdo = new PDO($dsn, DB_USER, DB_PASS);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // --- PROCESSAMENTO DO FORMULÁRIO (SALVAR/ATUALIZAR) ---
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['pontuacao'])) {
        $pontuacoes = $_POST['pontuacao'];
        $pdo->beginTransaction();

        foreach ($pontuacoes as $indicador_num => $pontuacao) {
            $pontuacao_limpa = floatval(str_replace(',', '.', $pontuacao));
            
            // Usando ON DUPLICATE KEY UPDATE para inserir ou atualizar
            $stmt = $pdo->prepare("
                INSERT INTO avaliacao_indicadores 
                    (usuario_id, ano, trimestre, indicador_num, pontuacao)
                VALUES 
                    (:usuario_id, :ano, :trimestre, :indicador_num, :pontuacao)
                ON DUPLICATE KEY UPDATE
                    pontuacao = VALUES(pontuacao)
            ");

            $stmt->execute([
                'usuario_id' => $id_selecionado,
                'ano' => $ano_lectivo,
                'trimestre' => $trimestre_focado,
                'indicador_num' => $indicador_num,
                'pontuacao' => $pontuacao_limpa
            ]);
        }
        $pdo->commit();
        $success_message = "Pontuações salvas com sucesso para o {$trimestre_focado}º Trimestre de {$ano_lectivo}!";
    }
    
    // 2. Busca Dados do Funcionário
    $stmt_func = $pdo->prepare("SELECT nome, perfil AS cargo FROM usuarios WHERE id = ?");
    $stmt_func->execute([$id_selecionado]);
    $dados_funcionario = $stmt_func->fetch();
    if (!$dados_funcionario) { die("Erro: Funcionário com ID {$id_selecionado} não encontrado."); }

    // 3. Busca Pontuações Existentes para o Trimestre (para pré-preenchimento)
    $stmt_notas = $pdo->prepare("
        SELECT indicador_num, pontuacao 
        FROM avaliacao_indicadores 
        WHERE usuario_id = :id AND ano = :ano AND trimestre = :trimestre
    ");
    $stmt_notas->execute([
        'id' => $id_selecionado, 
        'ano' => $ano_lectivo, 
        'trimestre' => $trimestre_focado
    ]);
    
    $pontuacoes_existentes = [];
    while ($row = $stmt_notas->fetch()) {
        // Armazena a pontuação usando o número do indicador como chave
        $pontuacoes_existentes[$row['indicador_num']] = $row['pontuacao'];
    }


} catch (PDOException $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    $error_message = "Erro de Banco de Dados: " . $e->getMessage();
}

// Função auxiliar para formatar a pontuação (evita erros com float/vírgula)
function format_pontuacao($valor) {
    if (is_null($valor) || $valor === '') return '';
    // Formata o número para duas casas decimais com ponto (padrão de entrada)
    return number_format($valor, 2, '.', ''); 
}
?>

<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Avaliar Desempenho - <?= $dados_funcionario['nome'] ?? 'Funcionário' ?></title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background-color: #f4f4f4; }
        .container { max-width: 700px; margin: 0 auto; background: white; padding: 25px; border-radius: 8px; box-shadow: 0 0 10px rgba(0,0,0,0.1); }
        h1 { color: #004080; border-bottom: 2px solid #ccc; padding-bottom: 10px; }
        .info { background-color: #e9ecef; padding: 10px; border-radius: 5px; margin-bottom: 20px; }
        .info p { margin: 5px 0; }
        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        th, td { border: 1px solid #ddd; padding: 10px; text-align: left; }
        th { background-color: #004080; color: white; }
        input[type="number"] { width: 90%; padding: 8px; box-sizing: border-box; border: 1px solid #ccc; border-radius: 4px; text-align: center; }
        .btn-save { display: block; width: 100%; padding: 10px; background-color: #28a745; color: white; border: none; border-radius: 5px; cursor: pointer; font-size: 16px; margin-top: 20px; }
        .btn-save:hover { background-color: #218838; }
        .message-error { color: red; background-color: #fdd; padding: 10px; border: 1px solid red; margin-bottom: 15px; border-radius: 4px; }
        .message-success { color: green; background-color: #dfd; padding: 10px; border: 1px solid green; margin-bottom: 15px; border-radius: 4px; }
        .nav-trimestre a { padding: 8px 15px; text-decoration: none; border: 1px solid #ccc; margin-right: 5px; border-radius: 4px; }
        .nav-trimestre a.active { background-color: #007bff; color: white; border-color: #007bff; }
    </style>
</head>
<body>

<div class="container">
    <a href="lista_funcionarios.php" style="text-decoration: none;">&larr; Voltar à Lista</a>

    <h1>Avaliação de Desempenho</h1>

    <?php if ($error_message): ?>
        <div class="message-error"><?= nl2br(htmlspecialchars($error_message)) ?></div>
    <?php endif; ?>

    <?php if ($success_message): ?>
        <div class="message-success"><?= htmlspecialchars($success_message) ?></div>
    <?php endif; ?>

    <div class="info">
        <p><strong>Funcionário:</strong> <?= htmlspecialchars($dados_funcionario['nome'] ?? 'N/D') ?></p>
        <p><strong>Cargo:</strong> <?= htmlspecialchars($dados_funcionario['cargo'] ?? 'N/D') ?></p>
        <p><strong>Ano Lectivo:</strong> <?= htmlspecialchars($ano_lectivo) ?></p>
    </div>

    <div class="nav-trimestre">
        <strong>Trimestre:</strong>
        <?php for ($t = 1; $t <= 3; $t++): ?>
            <a href="?id=<?= $id_selecionado ?>&ano=<?= $ano_lectivo ?>&trimestre=<?= $t ?>" 
               class="<?= $trimestre_focado == $t ? 'active' : '' ?>">
                <?= $t ?>º Trimestre
            </a>
        <?php endfor; ?>
    </div>
    <hr>
    
    <h2>Preencher <?= $trimestre_focado ?>º Trimestre</h2>

    <form method="POST">
        <table>
            <thead>
                <tr>
                    <th style="width: 10%;">Nº</th>
                    <th style="width: 70%;">Indicador de Desempenho</th>
                    <th style="width: 20%;">Pontuação (0-100)</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($indicadores_map as $num => $nome): ?>
                <tr>
                    <td style="text-align: center;"><?= $num ?></td>
                    <td><?= $nome ?></td>
                    <td style="text-align: center;">
                        <input 
                            type="number" 
                            name="pontuacao[<?= $num ?>]" 
                            min="0" 
                            max="100" 
                            step="0.01"
                            value="<?= format_pontuacao($pontuacoes_existentes[$num] ?? '') ?>"
                            required
                        >
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <button type="submit" class="btn-save">💾 Salvar Avaliação do Trimestre</button>
    </form>
    
    <div style="margin-top: 20px;">
        <a href="documento_oficial.php?id=<?= $id_selecionado ?>&modo=trimestral&trimestre=<?= $trimestre_focado ?>" 
           style="color: #007bff; text-decoration: none;">
           ▶️ Ver Ficha Trimestral (Gerada)
        </a>
    </div>

</div>

</body>
</html>