<?php
// Inicia a sessão para acessar os dados do aluno logado.
session_start();
include 'conexao.php';

// Verifica se o aluno está logado e se o ID da turma está na sessão.
if (!isset($_SESSION['aluno']) || !isset($_SESSION['aluno']['turma_id'])) {
    header("Location: login_aluno.php");
    exit;
}

// Obtém o ID da turma do aluno logado.
$turma_id = $_SESSION['aluno']['turma_id'];
$aluno_logado = $_SESSION['aluno'];

// Prepara e executa a consulta para buscar todos os alunos da turma, ordenando pelo nome.
// Uso de prepared statement para segurança contra SQL injection.
$stmt = $conn->prepare("SELECT * FROM alunos WHERE turma_id = ? ORDER BY nome ASC");
$stmt->bind_param("i", $turma_id);
$stmt->execute();
$result = $stmt->get_result();

$alunos = [];
while ($row = $result->fetch_assoc()) {
    $alunos[] = $row;
}
?>

<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <title>Lista da Turma</title>
    <!-- Adiciona a meta tag viewport para compatibilidade com telemóveis. -->
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <!-- Inclui a fonte FontAwesome para ícones. -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        /* Estilos base e mobile-first */
        body { 
            margin: 0; 
            font-family: 'Segoe UI', sans-serif; 
            background: #f4f6f9; 
            display: flex; 
            flex-direction: column;
            min-height: 100vh;
        }

        /* --- Barra Superior (Header) --- */
        .topbar {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 60px;
            background: #003366;
            color: white;
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 0 15px;
            z-index: 1000;
            box-shadow: 0 2px 5px rgba(0,0,0,0.2);
        }
        .menu-toggle {
            cursor: pointer;
            font-size: 24px;
        }
        .user-info {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .user-info img {
            width: 40px; 
            height: 40px;
            border-radius: 50%;
            object-fit: cover;
            border: 2px solid white;
        }

        /* --- Barra Lateral (Sidebar) --- */
        .sidebar {
            width: 250px;
            height: 100vh;
            background: #003366;
            color: white;
            position: fixed;
            top: 0; 
            left: 0;
            padding-top: 60px;
            transform: translateX(-100%);
            transition: transform 0.3s ease-in-out;
            z-index: 999;
            box-shadow: 2px 0 5px rgba(0,0,0,0.2);
            overflow-y: auto;
        }
        .sidebar.active {
            transform: translateX(0);
        }
        .sidebar a {
            color: white;
            padding: 15px 20px;
            display: block;
            text-decoration: none;
            border-bottom: 1px solid rgba(255,255,255,0.1);
            transition: background 0.2s;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .sidebar a:hover {
            background: #0056b3;
        }

        /* --- Conteúdo Principal --- */
        .main-content {
            margin-top: 60px;
            padding: 20px;
            flex-grow: 1; 
            transition: margin-left 0.3s;
        }
        .table-container {
            background: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0,0,0,0.1);
            overflow-x: auto;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            font-size: 0.9em;
        }
        th, td {
            text-align: left;
            padding: 12px 15px;
            border-bottom: 1px solid #ddd;
        }
        th {
            background-color: #003366;
            color: white;
            font-weight: bold;
        }
        tr:hover {
            background-color: #f1f1f1;
        }
        .back-link {
            display: inline-block;
            margin-bottom: 20px;
            text-decoration: none;
            color: #0056b3;
            font-weight: bold;
        }
        .back-link i {
            margin-right: 5px;
        }

        /* --- Estilos para Desktop --- */
        @media (min-width: 768px) {
            body { display: flex; flex-direction: row; }
            .sidebar {
                transform: translateX(0);
                position: relative;
            }
            .topbar {
                left: 250px;
                width: calc(100% - 250px);
            }
            .main-content {
                margin-left: 0;
                width: calc(100% - 250px);
            }
            .menu-toggle {
                display: none;
            }
        }
    </style>
</head>
<body>
    <!-- Barra Superior -->
    <div class="topbar">
        <!-- Botão de menu "hambúrguer" para telemóveis -->
        <div class="menu-toggle" onclick="toggleSidebar()">☰</div>
        <div><strong>EMILFRICKBG0026 | Lista da Turma</strong></div>
        <div class="user-info">
            <span><?= htmlspecialchars($aluno_logado['nome']) ?></span>
            <img src="fotos/<?= htmlspecialchars($aluno_logado['foto'] ?? 'sem-foto.png') ?>" alt="Foto do aluno">
        </div>
    </div>

    <!-- Barra lateral -->
    <div class="sidebar" id="sidebar">
        <a href="painel_aluno.php"><i class="fas fa-home"></i> Início</a>
        <a href="ver_lista_aluno.php"><i class="fas fa-users"></i> Ver Lista da Turma</a>
        <a href="ver_boletim_aluno.php"><i class="fas fa-file-alt"></i> Ver Boletim</a>
        <a href="consultar_propinas_alunos.php" class="menu-item"><i class="fas fa-upload"></i> Consultar Propinas</a>
        <a href="trabalhos_turma_aluno.php"><i class="fas fa-folder-open"></i> Ver Trabalhos</a>
        <a href="alterar_senha_aluno.php"><i class="fas fa-key"></i> Alterar Senha</a>
        <a href="enviar_documento1.php" class="menu-item"><i class="fas fa-upload"></i> Enviar Documento</a>
        <a href="pauta_final_aluno.php" class="menu-item"><i class="fas fa-list"></i> Termo de Frequência</a>
        <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Sair</a>
    </div>

    <!-- Conteúdo principal -->
    <div class="main-content">
        <!-- O botão/link para voltar ao painel principal -->
        <a href="painel_aluno.php" class="back-link"><i class="fas fa-arrow-left"></i> Voltar ao Painel</a>
        <div class="table-container">
            <h2>Lista de Alunos da Turma</h2>
            <?php if (empty($alunos)): ?>
                <p>Nenhum aluno encontrado nesta turma.</p>
            <?php else: ?>
                <table>
                    <thead>
                        <tr>
                            <th>Nº</th>
                            <th>Nome</th>
                            <th>Curso</th>
                            <th>Classe</th>
                            <th>Turma</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php $contador = 1; ?>
                        <?php foreach ($alunos as $aluno): ?>
                            <tr>
                                <td><?= $contador++ ?></td>
                                <td><?= htmlspecialchars($aluno['nome']) ?></td>
                                <td><?= htmlspecialchars($aluno['curso'] ?? '---') ?></td>
                                <td><?= htmlspecialchars($aluno['classe'] ?? '---') ?></td>
                                <td><?= htmlspecialchars($aluno['turma_nome'] ?? '---') ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>

    <script>
        // Função para alternar a exibição da barra lateral em ecrãs pequenos.
        function toggleSidebar() {
            const sidebar = document.getElementById('sidebar');
            sidebar.classList.toggle('active');
        }
    </script>
</body>
</html>